/*

  t1area
  ======

  Scan a Type 1 font and tell the area of each character.
  Also tell how much percent of the bounding box of the character
  is filled by this area.

  Copyright (c) 2003 Thomas Baruchel
  
  Permission is hereby granted, free of charge, to any person obtaining
  a copy of this software and associated documentation files (the
  "Software"), to deal in the Software without restriction, including
  without limitation the rights to use, copy, modify, merge, publish,
  distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to
  the following conditions:
  
  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.
  
  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
  CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
  TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
     SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include <stdio.h>
#include <stdlib.h>
#include <t1lib.h>
#include <math.h>

int usage(void) {
  fprintf(stderr,"Usage: t1area [-v] file\n    -v  verbose output\n");
  return(EXIT_FAILURE);
}

double integrale(double t1, double t2, int ay, int by, int cy, int dy,
                 int a, int b, int c) {
  return(
  ( ay*a*t2*t2*t2*t2*t2*t2/6
    + (ay*b+by*a)*t2*t2*t2*t2*t2/5
    + (ay*c+by*b+cy*a)*t2*t2*t2*t2/4
    + (by*c+cy*b+dy*a)*t2*t2*t2/3
    + (cy*c+dy*b)*t2*t2/2
    + dy*c*t2 )
-
  ( ay*a*t1*t1*t1*t1*t1*t1/6
    + (ay*b+by*a)*t1*t1*t1*t1*t1/5
    + (ay*c+by*b+cy*a)*t1*t1*t1*t1/4
    + (by*c+cy*b+dy*a)*t1*t1*t1/3
    + (cy*c+dy*b)*t1*t1/2
    + dy*c*t1 )
  );
}

void area(T1_OUTLINE *outline, BBox bb, char *name, unsigned char v) {
  auto int xpos = 0;
  auto int ypos = -bb.lly;
  auto double area = 0;
  do {
    switch(((T1_PATHSEGMENT *) outline)->type) {
      case T1_PATHTYPE_MOVE: {
        xpos += (int) T1_NEARESTPOINT(((T1_PATHSEGMENT *) outline)->dest.x);
        ypos -= (int) T1_NEARESTPOINT(((T1_PATHSEGMENT *) outline)->dest.y);
        break;
      }
      case T1_PATHTYPE_LINE: {
        auto int dx =
          (int) T1_NEARESTPOINT(((T1_PATHSEGMENT *) outline)->dest.x);
        auto int dy =
          - (int) T1_NEARESTPOINT(((T1_PATHSEGMENT *) outline)->dest.y);
        auto double surface = (double) (abs(dx)*((dy<0) ? (ypos+dy) : ypos))
                  + ((double) (abs(dy*dx)))/((double) 2);
        area += (dx<0) ? surface : -surface;
        xpos += dx;
        ypos += dy;
        break;
      }
      case T1_PATHTYPE_BEZIER: {
        auto int x0 = xpos;
        auto int y0 = ypos;
        auto int x1 = 
          xpos+T1_NEARESTPOINT(((T1_BEZIERSEGMENT *) outline)->B.x);
        auto int y1 = 
          ypos-T1_NEARESTPOINT(((T1_BEZIERSEGMENT *) outline)->B.y);
        auto int x2 = 
          xpos+T1_NEARESTPOINT(((T1_BEZIERSEGMENT *) outline)->C.x);
        auto int y2 = 
          ypos-T1_NEARESTPOINT(((T1_BEZIERSEGMENT *) outline)->C.y);
        /* x'(t) = a.t^2 + b.t + c */
        auto int a,b,c;
        /* y(t) = ay.t^3 + by.t^2 + cy.t + dy */
        auto int ay,by,cy;
        /* auto int dy; useless */
        xpos += (int) T1_NEARESTPOINT(outline->dest.x);
        ypos -= (int) T1_NEARESTPOINT(outline->dest.y);
        ay = -y0 + 3*y1 -3*y2 + ypos;
        by = 3*y0-6*y1+3*y2;
        cy = -3*y0+3*y1;
        /* dy = y0; useless */
        a = 3*(-x0+3*x1-3*x2+xpos);
        b = 6*(x0-2*x1+x2);
        c = 3*(-x0+x1);
        area -= integrale(0,1,ay,by,cy,y0,a,b,c);
        break;
      }
    }
  } while ((outline = outline->link) != NULL);
  if(area != (double) 0) {
    fprintf(stdout,"/%s    area = %.2f",name,area);
    if(v!=0) {
      fprintf(stdout," (%.2f %% of BBox)",
        area/((bb.urx-bb.llx)*(bb.ury-bb.lly))*100);
    }
    fprintf(stdout,"\n");
  }
}

int main(int argc, char **argv)
{
  static signed int i;
  static unsigned char j=0;
  static char *fontname = NULL;
  static unsigned char v = 0;

  for(i=1;i<argc;i++) {
    if((argv[i][0]=='-')
        && (argv[i][1]=='v')
        && (argv[i][2]=='\0')) v = 1;
    else if(fontname==NULL) fontname = argv[i];
      else exit(usage());
  }
  if(fontname==NULL) exit(usage());
  if (T1_InitLib(NO_LOGFILE)==NULL) {
    fprintf(stderr, "Initialization of t1lib failed\n");
    exit(EXIT_FAILURE);
  }
  i = T1_AddFont(fontname);
  if(i<0) {
    T1_CloseLib();
    if(i==-1) {
      fprintf(stderr, "Font file could not be located\n");
      exit(EXIT_FAILURE);
    }
    fprintf(stderr, "Memory allocation failure\n");
    exit(EXIT_FAILURE);
  }
  if(T1_LoadFont(i)==-1) {
    T1_CloseLib();
    fprintf(stderr, "t1lib could not load the font\n");
    exit(EXIT_FAILURE);
  }
  do {
    auto T1_OUTLINE *myoutline;
    area(myoutline = T1_GetCharOutline(i,j,1000,NULL),
                     T1_GetCharBBox(i,j),
                     T1_GetCharName(i,j),v);
    T1_FreeOutline(myoutline);
  } while(j++<255);
  T1_CloseLib();
  return(EXIT_SUCCESS);
}
