package org.unicode.cldr.tool;

import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.ImmutableSortedSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.TreeMultimap;
import com.ibm.icu.impl.Relation;
import com.ibm.icu.impl.Row;
import com.ibm.icu.impl.Row.R2;
import com.ibm.icu.text.UnicodeSet;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;
import org.unicode.cldr.util.CLDRConfig;
import org.unicode.cldr.util.CLDRFile;
import org.unicode.cldr.util.CLDRFile.DraftStatus;
import org.unicode.cldr.util.CLDRPaths;
import org.unicode.cldr.util.CalculatedCoverageLevels;
import org.unicode.cldr.util.CldrUtility;
import org.unicode.cldr.util.CoverageInfo;
import org.unicode.cldr.util.Factory;
import org.unicode.cldr.util.Level;
import org.unicode.cldr.util.LsrvCanonicalizer;
import org.unicode.cldr.util.LsrvCanonicalizer.TestDataTypes;
import org.unicode.cldr.util.NameGetter;
import org.unicode.cldr.util.StandardCodes.LstrType;
import org.unicode.cldr.util.SupplementalDataInfo;
import org.unicode.cldr.util.TempPrintWriter;

public class GenerateLocaleIDTestData {

    private static final LsrvCanonicalizer rrs = LsrvCanonicalizer.getInstance();
    private static final CLDRConfig CLDR_CONFIG = CLDRConfig.getInstance();
    private static final CLDRFile ENGLISH = CLDR_CONFIG.getEnglish();

    private static final String DO_NOT_EDIT =
            "# DO NOT EDIT THIS FILE, instead regenerate it using "
                    + GenerateLocaleIDTestData.class.getSimpleName()
                    + ".java";

    public static void main(String[] args) throws IOException {
        // localeCanonicalization.txt

        try (TempPrintWriter pw =
                TempPrintWriter.openUTF8Writer(
                        CLDRPaths.TEST_DATA + "localeIdentifiers", "localeCanonicalization.txt")) {
            pw.println(DO_NOT_EDIT);
            pw.println("# Test data for locale identifier canonicalization");
            pw.println(CldrUtility.getCopyrightString("#  "));
            pw.println(
                    "#\n"
                            + "# Format:\n"
                            + "# <source locale identifier>\t;\t<expected canonicalized locale identifier>\n"
                            + "#\n"
                            + "# The data lines are divided into 4 sets:\n"
                            + "#   "
                            + LsrvCanonicalizer.TestDataTypes.explicit
                            + ":    a short list of explicit test cases.\n"
                            + "#   "
                            + LsrvCanonicalizer.TestDataTypes.fromAliases
                            + ": test cases generated from the alias data.\n"
                            + "#   "
                            + LsrvCanonicalizer.TestDataTypes.decanonicalized
                            + ": test cases generated by reversing the normalization process.\n"
                            + "#   "
                            + LsrvCanonicalizer.TestDataTypes.withIrrelevants
                            + ": test cases generated from the others by adding irrelevant fields where possible,\n"
                            + "#                           to ensure that the canonicalization implementation is not sensitive to irrelevant fields. These include:\n"
                            + "#     Language: "
                            + rrs.getIrrelevantField(LstrType.language)
                            + "\n"
                            + "#     Script:   "
                            + rrs.getIrrelevantField(LstrType.script)
                            + "\n"
                            + "#     Region:   "
                            + rrs.getIrrelevantField(LstrType.region)
                            + "\n"
                            + "#     Variant:  "
                            + rrs.getIrrelevantField(LstrType.variant)
                            + "\n"
                            + "######\n\n");
            for (Entry<TestDataTypes, Map<String, String>> mainEntry :
                    rrs.getTestData(null).entrySet()) {
                TestDataTypes type = mainEntry.getKey();
                pw.println("\n# " + type + "\n");
                for (Entry<String, String> entry : mainEntry.getValue().entrySet()) {
                    String toTest = entry.getKey();
                    String expected = entry.getValue();
                    pw.println(toTest + "\t;\t" + expected);
                }
            }
        }

        // localeDisplayName.txt

        String[] testInputLocales = {
            "es",
            "es-419",
            "es-Cyrl-MX",
            "hi-Latn",
            "nl-BE",
            "nl-Latn-BE",
            "en-MM",
            "zh-Hans-fonipa"
        };
        try (TempPrintWriter pw =
                TempPrintWriter.openUTF8Writer(
                        CLDRPaths.TEST_DATA + "localeIdentifiers", "localeDisplayName.txt")) {
            pw.println(DO_NOT_EDIT);
            pw.println(
                    "# Test data for locale display name generation\n"
                            + CldrUtility.getCopyrightString("#  ")
                            + "\n# Format:\n"
                            + "# @locale=<locale to display in>\n"
                            + "# @languageDisplay=[standard|dialect]\n"
                            + "#    standard - always display additional subtags like region in parentheses\n"
                            + "#    dialect - form compounds like \"Flemish\" for nl_BE\n"
                            + "# <locale to display> ; <expected display name>\n"
                            + "\n"
                            + "@locale=en\n"
                            + "@languageDisplay=standard\n");
            pw.println("\n# Simple cases: Language, script, region, variants\n");
            showDisplayNames(pw, ENGLISH, NameGetter.NameOpt.COMPOUND_ONLY, testInputLocales);
            pw.println(
                    "\n#Note that the order of the variants is alphabetized before generating names\n");
            showDisplayNames(
                    pw, ENGLISH, NameGetter.NameOpt.COMPOUND_ONLY, "en-Latn-GB-scouse-fonipa");
            pw.println("\n# Add extensions, and verify their order\n");
            showDisplayNames(
                    pw,
                    ENGLISH,
                    NameGetter.NameOpt.COMPOUND_ONLY,
                    "en-u-nu-thai-ca-islamic-civil",
                    "hi-u-nu-latn-t-en-h0-hybrid",
                    "en-u-nu-deva-t-de");
            pw.println("\n# Test ordering of extensions (include well-formed but invalid cases)\n");
            showDisplayNames(
                    pw,
                    ENGLISH,
                    NameGetter.NameOpt.COMPOUND_ONLY,
                    "fr-z-zz-zzz-v-vv-vvv-u-uu-uuu-t-ru-Cyrl-s-ss-sss-a-aa-aaa-x-u-x");

            pw.println(
                    "\n# Comprehensive list (mostly comprehensive: currencies, subdivisions, timezones have abbreviated lists)\n");
            SupplementalDataInfo SDI = CLDR_CONFIG.getSupplementalDataInfo();
            Relation<String, String> extensionToKeys = SDI.getBcp47Extension2Keys();
            Multimap<String, String> keyToExtensions = TreeMultimap.create();
            for (Entry<String, String> entry : extensionToKeys.entrySet()) {
                keyToExtensions.put(entry.getValue(), entry.getKey());
            }
            final Relation<String, String> keyToValues = SDI.getBcp47Keys();
            Map<R2<String, String>, String> deprecated = SDI.getBcp47Deprecated();

            ImmutableMultimap<String, String> overrides =
                    ImmutableMultimap.<String, String>builder()
                            .putAll("cu", "eur", "jpy", "usd", "chf")
                            .putAll("rg", "gbsct", "gbeng")
                            .putAll("sd", "gbsct", "gbwls")
                            .putAll("tz", "uslax", "gblon", "chzrh")
                            .putAll("dx", "thai")
                            .putAll("vt", "abcd")
                            .putAll("x0", "foobar2")
                            .putAll(
                                    "kr",
                                    "arab",
                                    "digit-deva-latn",
                                    "currency",
                                    "digit",
                                    "punct",
                                    "space",
                                    "symbol")
                            .build();

            final UnicodeSet upper = new UnicodeSet("[A-Z]").freeze();

            for (String key : keyToValues.keySet()) {
                if ("true".equals(deprecated.get(Row.of(key, "")))) {
                    continue;
                }
                for (String extension : keyToExtensions.get(key)) {
                    Collection<String> values =
                            overrides.containsKey(key)
                                    ? overrides.get(key)
                                    : ImmutableSortedSet.copyOf(keyToValues.get(key));
                    for (String value : values) {
                        if ("true".equals(deprecated.get(Row.of(key, value)))) {
                            continue;
                        }
                        final String sampleLocale = "en-" + extension + "-" + key + "-" + value;
                        if (upper.containsSome(value)) {
                            System.err.println("** FIX NAME: " + sampleLocale);
                        } else {
                            showDisplayNames(
                                    pw, ENGLISH, NameGetter.NameOpt.COMPOUND_ONLY, sampleLocale);
                        }
                    }
                }
            }

            pw.println();

            Factory factory = CLDR_CONFIG.getCldrFactory();
            CoverageInfo coverageInfo = CLDR_CONFIG.getCoverageInfo();
            for (String locale : factory.getAvailableLanguages()) {
                for (NameGetter.NameOpt nameOpt :
                        List.of(NameGetter.NameOpt.COMPOUND_ONLY, NameGetter.NameOpt.DEFAULT)) {
                    CLDRFile cldrFile =
                            factory.make(
                                    locale,
                                    true,
                                    DraftStatus.contributed); // don't include provisional data

                    // This is the CLDR "effective coverage level"
                    Level coverageLevel =
                            CalculatedCoverageLevels.getInstance()
                                    .getEffectiveCoverageLevel(locale);

                    if (coverageLevel == null || !coverageLevel.isAtLeast(Level.MODERN)) {
                        continue;
                    }

                    Map<String, String> displayNames =
                            prepareDisplayNames(cldrFile, nameOpt, testInputLocales);

                    pw.println("\n@locale=" + locale);
                    String languageDisplayVal =
                            nameOpt == NameGetter.NameOpt.COMPOUND_ONLY ? "standard" : "dialect";
                    pw.println("@languageDisplay=" + languageDisplayVal + "\n");

                    showDisplayNames(pw, displayNames);
                    pw.println();
                }
            }
        }
    }

    private static Map<String, String> prepareDisplayNames(
            CLDRFile formattingLocaleFile, NameGetter.NameOpt nameOpt, String... locales) {
        Map<String, String> displayNames = new TreeMap<>();
        for (String locale : locales) {
            String name =
                    formattingLocaleFile.nameGetter().getNameFromIdentifierOpt(locale, nameOpt);
            if (name.contains("null")) {
                System.err.println("** REPLACE: " + locale + "; " + name);
            } else {
                displayNames.put(locale, name);
            }
        }

        if (displayNames.isEmpty()) {
            return null;
        }
        return displayNames;
    }

    private static void showDisplayNames(
            TempPrintWriter pw,
            CLDRFile formattingLocaleFile,
            NameGetter.NameOpt nameOpt,
            String... locales) {
        showDisplayNames(pw, prepareDisplayNames(formattingLocaleFile, nameOpt, locales));
    }

    private static void showDisplayNames(TempPrintWriter pw, Map<String, String> displayNames) {
        for (Entry<String, String> entry : displayNames.entrySet()) {
            pw.println(entry.getKey() + "; " + entry.getValue());
        }
    }
}
