/*
 * Copyright (c) 2019-2020 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#pragma once

#ifdef __ARM_FEATURE_SVE

template<>
void MergeResults<3, 8, true>(__fp16 *out, const __fp16 *in, const int ldout, const int y0, const int ymax, const int x0, const int xmax, const __fp16 *bias, Activation act, bool append)
{
    const __fp16 *inptr = in;
    __fp16 nullbias[384];
    __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());
    __fp16 maxval =   static_cast<__fp16>(std::numeric_limits<float>::infinity());

    switch(act.type)
    {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0.0f;
            break;
    }

    if (!append && !bias)
    {
        memset(nullbias, 0, (3 * get_vector_length<__fp16>() * sizeof(__fp16)));
    }

    for (int y=y0; y<ymax; y+=8)
    {
        __fp16 *outptr0 = out + (y * ldout) + x0;
        __fp16 *outptr1 = outptr0 + ldout;
        __fp16 *outptr2 = outptr1 + ldout;
        __fp16 *outptr3 = outptr2 + ldout;
        __fp16 *outptr4 = outptr3 + ldout;
        __fp16 *outptr5 = outptr4 + ldout;
        __fp16 *outptr6 = outptr5 + ldout;
        __fp16 *outptr7 = outptr6 + ldout;

        const int height = ymax - y;

        for (int i=x0; i<xmax; i+=(3 * get_vector_length<__fp16>()))
        {
            if (append)
            {
                switch(height)
                {
                case 1:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PLDL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[outptr0]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z10.h, p0/z, [%[inptr]]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z3.h, p1/z, [%[outptr0], #1, MUL VL]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "ld1h z11.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "fmin z10.h, p0/m, z10.h, z0.h\n"
                            "ld1h z4.h, p2/z, [%[outptr0], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z12.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmax z10.h, p0/m, z10.h, z1.h\n"
                            "fmin z11.h, p1/m, z11.h, z0.h\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "st1h z10.h, p0, [%[outptr0]]\n"
                            "fmax z11.h, p1/m, z11.h, z1.h\n"
                            "fmin z12.h, p2/m, z12.h, z0.h\n"
                            "st1h z11.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fmax z12.h, p2/m, z12.h, z1.h\n"
                            "st1h z12.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 2:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PLDL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[outptr0]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z10.h, p0/z, [%[inptr]]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z5.h, p0/z, [%[outptr1]]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "ld1h z3.h, p1/z, [%[outptr0], #1, MUL VL]\n"
                            "ld1h z11.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z13.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr1], #0x60]\n"
                            "fmin z10.h, p0/m, z10.h, z0.h\n"
                            "ld1h z4.h, p2/z, [%[outptr0], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z12.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z6.h, p1/z, [%[outptr1], #1, MUL VL]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "fmax z10.h, p0/m, z10.h, z1.h\n"
                            "ld1h z7.h, p2/z, [%[outptr1], #2, MUL VL]\n"
                            "fmin z11.h, p1/m, z11.h, z0.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "st1h z10.h, p0, [%[outptr0]]\n"
                            "fmax z11.h, p1/m, z11.h, z1.h\n"
                            "fmin z12.h, p2/m, z12.h, z0.h\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "st1h z11.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "fmax z12.h, p2/m, z12.h, z1.h\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z12.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "st1h z13.h, p0, [%[outptr1]]\n"
                            "st1h z14.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "st1h z15.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 3:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PLDL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[outptr0]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z10.h, p0/z, [%[inptr]]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z5.h, p0/z, [%[outptr1]]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "ld1h z3.h, p1/z, [%[outptr0], #1, MUL VL]\n"
                            "ld1h z11.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z13.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr1], #0x60]\n"
                            "fmin z10.h, p0/m, z10.h, z0.h\n"
                            "ld1h z4.h, p2/z, [%[outptr0], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z12.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z6.h, p1/z, [%[outptr1], #1, MUL VL]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fmax z10.h, p0/m, z10.h, z1.h\n"
                            "ld1h z7.h, p2/z, [%[outptr1], #2, MUL VL]\n"
                            "fmin z11.h, p1/m, z11.h, z0.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z8.h, p0/z, [%[outptr2]]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "st1h z10.h, p0, [%[outptr0]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "ld1h z16.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fmax z11.h, p1/m, z11.h, z1.h\n"
                            "ld1h z9.h, p1/z, [%[outptr2], #1, MUL VL]\n"
                            "fmin z12.h, p2/m, z12.h, z0.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "ld1h z2.h, p2/z, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "st1h z11.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z10.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fmax z12.h, p2/m, z12.h, z1.h\n"
                            "prfm PLDL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z12.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "st1h z13.h, p0, [%[outptr1]]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z14.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "st1h z15.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fmin z10.h, p2/m, z10.h, z0.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z16.h, p0, [%[outptr2]]\n"
                            "fmax z10.h, p2/m, z10.h, z1.h\n"
                            "st1h z17.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "st1h z10.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 4:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PLDL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[outptr0]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z10.h, p0/z, [%[inptr]]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z5.h, p0/z, [%[outptr1]]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "ld1h z3.h, p1/z, [%[outptr0], #1, MUL VL]\n"
                            "ld1h z11.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z13.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr1], #0x60]\n"
                            "fmin z10.h, p0/m, z10.h, z0.h\n"
                            "ld1h z4.h, p2/z, [%[outptr0], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z12.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z6.h, p1/z, [%[outptr1], #1, MUL VL]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fmax z10.h, p0/m, z10.h, z1.h\n"
                            "ld1h z7.h, p2/z, [%[outptr1], #2, MUL VL]\n"
                            "fmin z11.h, p1/m, z11.h, z0.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z8.h, p0/z, [%[outptr2]]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "st1h z10.h, p0, [%[outptr0]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "ld1h z16.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fmax z11.h, p1/m, z11.h, z1.h\n"
                            "ld1h z9.h, p1/z, [%[outptr2], #1, MUL VL]\n"
                            "fmin z12.h, p2/m, z12.h, z0.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "ld1h z2.h, p2/z, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "st1h z11.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z10.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fmax z12.h, p2/m, z12.h, z1.h\n"
                            "ld1h z3.h, p0/z, [%[outptr3]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "ld1h z11.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "ld1h z4.h, p1/z, [%[outptr3], #1, MUL VL]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z12.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "ld1h z12.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z5.h, p2/z, [%[outptr3], #2, MUL VL]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z13.h, p0, [%[outptr1]]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "ld1h z13.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmin z10.h, p2/m, z10.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z15.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmax z10.h, p2/m, z10.h, z1.h\n"
                            "st1h z16.h, p0, [%[outptr2]]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "prfm PLDL1KEEP, [%[outptr3], #0x60]\n"
                            "fmin z11.h, p0/m, z11.h, z0.h\n"
                            "st1h z17.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmin z12.h, p1/m, z12.h, z0.h\n"
                            "st1h z10.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fmax z11.h, p0/m, z11.h, z1.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "fmax z12.h, p1/m, z12.h, z1.h\n"
                            "st1h z11.h, p0, [%[outptr3]]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "st1h z12.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "st1h z13.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 5:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PLDL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[outptr0]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z10.h, p0/z, [%[inptr]]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z5.h, p0/z, [%[outptr1]]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "ld1h z3.h, p1/z, [%[outptr0], #1, MUL VL]\n"
                            "ld1h z11.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z13.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr1], #0x60]\n"
                            "fmin z10.h, p0/m, z10.h, z0.h\n"
                            "ld1h z4.h, p2/z, [%[outptr0], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z12.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z6.h, p1/z, [%[outptr1], #1, MUL VL]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fmax z10.h, p0/m, z10.h, z1.h\n"
                            "ld1h z7.h, p2/z, [%[outptr1], #2, MUL VL]\n"
                            "fmin z11.h, p1/m, z11.h, z0.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z8.h, p0/z, [%[outptr2]]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "st1h z10.h, p0, [%[outptr0]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "ld1h z16.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fmax z11.h, p1/m, z11.h, z1.h\n"
                            "ld1h z9.h, p1/z, [%[outptr2], #1, MUL VL]\n"
                            "fmin z12.h, p2/m, z12.h, z0.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "ld1h z2.h, p2/z, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "st1h z11.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z10.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fmax z12.h, p2/m, z12.h, z1.h\n"
                            "ld1h z3.h, p0/z, [%[outptr3]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "ld1h z11.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "ld1h z4.h, p1/z, [%[outptr3], #1, MUL VL]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z12.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "ld1h z12.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z5.h, p2/z, [%[outptr3], #2, MUL VL]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z13.h, p0, [%[outptr1]]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "ld1h z13.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "ld1h z6.h, p0/z, [%[outptr4]]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmin z10.h, p2/m, z10.h, z0.h\n"
                            "ld1h z14.h, p0/z, [x8, #-4, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z7.h, p1/z, [%[outptr4], #1, MUL VL]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z15.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z15.h, p1/z, [x8, #-3, MUL VL]\n"
                            "fmax z10.h, p2/m, z10.h, z1.h\n"
                            "ld1h z8.h, p2/z, [%[outptr4], #2, MUL VL]\n"
                            "fmin z11.h, p0/m, z11.h, z0.h\n"
                            "st1h z16.h, p0, [%[outptr2]]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z16.h, p2/z, [x8, #-2, MUL VL]\n"
                            "fmin z12.h, p1/m, z12.h, z0.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmax z11.h, p0/m, z11.h, z1.h\n"
                            "st1h z17.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmax z12.h, p1/m, z12.h, z1.h\n"
                            "st1h z10.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "prfm PLDL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "st1h z11.h, p0, [%[outptr3]]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "st1h z12.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "prfm PLDL1KEEP, [%[outptr3], #0x60]\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "st1h z13.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x240]\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "st1h z14.h, p0, [%[outptr4]]\n"
                            "prfm PLDL1KEEP, [%[outptr4], #0x60]\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "st1h z15.h, p1, [%[outptr4], #1, MUL VL]\n"
                            "st1h z16.h, p2, [%[outptr4], #2, MUL VL]\n"
                            "addvl %[outptr4], %[outptr4], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 6:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PLDL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[outptr0]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z10.h, p0/z, [%[inptr]]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z5.h, p0/z, [%[outptr1]]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "ld1h z3.h, p1/z, [%[outptr0], #1, MUL VL]\n"
                            "ld1h z11.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z13.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr1], #0x60]\n"
                            "fmin z10.h, p0/m, z10.h, z0.h\n"
                            "ld1h z4.h, p2/z, [%[outptr0], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z12.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z6.h, p1/z, [%[outptr1], #1, MUL VL]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fmax z10.h, p0/m, z10.h, z1.h\n"
                            "ld1h z7.h, p2/z, [%[outptr1], #2, MUL VL]\n"
                            "fmin z11.h, p1/m, z11.h, z0.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z8.h, p0/z, [%[outptr2]]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "st1h z10.h, p0, [%[outptr0]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "ld1h z16.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fmax z11.h, p1/m, z11.h, z1.h\n"
                            "ld1h z9.h, p1/z, [%[outptr2], #1, MUL VL]\n"
                            "fmin z12.h, p2/m, z12.h, z0.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "ld1h z2.h, p2/z, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "st1h z11.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z10.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fmax z12.h, p2/m, z12.h, z1.h\n"
                            "ld1h z3.h, p0/z, [%[outptr3]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "ld1h z11.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "ld1h z4.h, p1/z, [%[outptr3], #1, MUL VL]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z12.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "ld1h z12.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z5.h, p2/z, [%[outptr3], #2, MUL VL]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z13.h, p0, [%[outptr1]]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "ld1h z13.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "ld1h z6.h, p0/z, [%[outptr4]]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmin z10.h, p2/m, z10.h, z0.h\n"
                            "ld1h z14.h, p0/z, [x8, #-4, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z7.h, p1/z, [%[outptr4], #1, MUL VL]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z15.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z15.h, p1/z, [x8, #-3, MUL VL]\n"
                            "fmax z10.h, p2/m, z10.h, z1.h\n"
                            "ld1h z8.h, p2/z, [%[outptr4], #2, MUL VL]\n"
                            "fmin z11.h, p0/m, z11.h, z0.h\n"
                            "st1h z16.h, p0, [%[outptr2]]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z16.h, p2/z, [x8, #-2, MUL VL]\n"
                            "fmin z12.h, p1/m, z12.h, z0.h\n"
                            "ld1h z9.h, p0/z, [%[outptr5]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "st1h z17.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fmax z11.h, p0/m, z11.h, z1.h\n"
                            "ld1h z17.h, p0/z, [x8, #-1, MUL VL]\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "ld1h z2.h, p1/z, [%[outptr5], #1, MUL VL]\n"
                            "fmax z12.h, p1/m, z12.h, z1.h\n"
                            "st1h z10.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "ld1h z10.h, p1/z, [x8]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z3.h, p2/z, [%[outptr5], #2, MUL VL]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "st1h z11.h, p0, [%[outptr3]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "ld1h z11.h, p2/z, [x8, #1, MUL VL]\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "st1h z12.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "st1h z13.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "prfm PLDL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "st1h z14.h, p0, [%[outptr4]]\n"
                            "fmin z17.h, p0/m, z17.h, z0.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z15.h, p1, [%[outptr4], #1, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "prfm PLDL1KEEP, [%[outptr3], #0x60]\n"
                            "fmax z17.h, p0/m, z17.h, z1.h\n"
                            "st1h z16.h, p2, [%[outptr4], #2, MUL VL]\n"
                            "fmin z10.h, p1/m, z10.h, z0.h\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                            "fmin z11.h, p2/m, z11.h, z0.h\n"
                            "st1h z17.h, p0, [%[outptr5]]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x240]\n"
                            "fmax z10.h, p1/m, z10.h, z1.h\n"
                            "prfm PLDL1KEEP, [%[outptr4], #0x60]\n"
                            "fmax z11.h, p2/m, z11.h, z1.h\n"
                            "addvl %[outptr4], %[outptr4], #3\n"
                            "st1h z10.h, p1, [%[outptr5], #1, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x280]\n"
                            "prfm PLDL1KEEP, [%[outptr5], #0x60]\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "st1h z11.h, p2, [%[outptr5], #2, MUL VL]\n"
                            "addvl %[outptr5], %[outptr5], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 7:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PLDL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[outptr0]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z10.h, p0/z, [%[inptr]]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z5.h, p0/z, [%[outptr1]]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "ld1h z3.h, p1/z, [%[outptr0], #1, MUL VL]\n"
                            "ld1h z11.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z13.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr1], #0x60]\n"
                            "fmin z10.h, p0/m, z10.h, z0.h\n"
                            "ld1h z4.h, p2/z, [%[outptr0], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z12.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z6.h, p1/z, [%[outptr1], #1, MUL VL]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fmax z10.h, p0/m, z10.h, z1.h\n"
                            "ld1h z7.h, p2/z, [%[outptr1], #2, MUL VL]\n"
                            "fmin z11.h, p1/m, z11.h, z0.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z8.h, p0/z, [%[outptr2]]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "st1h z10.h, p0, [%[outptr0]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "ld1h z16.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fmax z11.h, p1/m, z11.h, z1.h\n"
                            "ld1h z9.h, p1/z, [%[outptr2], #1, MUL VL]\n"
                            "fmin z12.h, p2/m, z12.h, z0.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "ld1h z2.h, p2/z, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "st1h z11.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z10.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fmax z12.h, p2/m, z12.h, z1.h\n"
                            "ld1h z3.h, p0/z, [%[outptr3]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "ld1h z11.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "ld1h z4.h, p1/z, [%[outptr3], #1, MUL VL]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z12.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "ld1h z12.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z5.h, p2/z, [%[outptr3], #2, MUL VL]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z13.h, p0, [%[outptr1]]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "ld1h z13.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "ld1h z6.h, p0/z, [%[outptr4]]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmin z10.h, p2/m, z10.h, z0.h\n"
                            "ld1h z14.h, p0/z, [x8, #-4, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z7.h, p1/z, [%[outptr4], #1, MUL VL]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z15.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z15.h, p1/z, [x8, #-3, MUL VL]\n"
                            "fmax z10.h, p2/m, z10.h, z1.h\n"
                            "ld1h z8.h, p2/z, [%[outptr4], #2, MUL VL]\n"
                            "fmin z11.h, p0/m, z11.h, z0.h\n"
                            "st1h z16.h, p0, [%[outptr2]]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z16.h, p2/z, [x8, #-2, MUL VL]\n"
                            "fmin z12.h, p1/m, z12.h, z0.h\n"
                            "ld1h z9.h, p0/z, [%[outptr5]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "st1h z17.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fmax z11.h, p0/m, z11.h, z1.h\n"
                            "ld1h z17.h, p0/z, [x8, #-1, MUL VL]\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "ld1h z2.h, p1/z, [%[outptr5], #1, MUL VL]\n"
                            "fmax z12.h, p1/m, z12.h, z1.h\n"
                            "st1h z10.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "ld1h z10.h, p1/z, [x8]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z3.h, p2/z, [%[outptr5], #2, MUL VL]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "st1h z11.h, p0, [%[outptr3]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "ld1h z11.h, p2/z, [x8, #1, MUL VL]\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "ld1h z4.h, p0/z, [%[outptr6]]\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "st1h z12.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "ld1h z12.h, p0/z, [x8, #2, MUL VL]\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "ld1h z5.h, p1/z, [%[outptr6], #1, MUL VL]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z13.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "ld1h z13.h, p1/z, [x8, #3, MUL VL]\n"
                            "fmin z17.h, p0/m, z17.h, z0.h\n"
                            "ld1h z6.h, p2/z, [%[outptr6], #2, MUL VL]\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "st1h z14.h, p0, [%[outptr4]]\n"
                            "fmin z10.h, p1/m, z10.h, z0.h\n"
                            "ld1h z14.h, p2/z, [x8, #4, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmax z17.h, p0/m, z17.h, z1.h\n"
                            "st1h z15.h, p1, [%[outptr4], #1, MUL VL]\n"
                            "fmax z10.h, p1/m, z10.h, z1.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmin z11.h, p2/m, z11.h, z0.h\n"
                            "st1h z16.h, p2, [%[outptr4], #2, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "prfm PLDL1KEEP, [%[outptr2], #0x60]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "st1h z17.h, p0, [%[outptr5]]\n"
                            "fmax z11.h, p2/m, z11.h, z1.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmin z12.h, p0/m, z12.h, z0.h\n"
                            "st1h z10.h, p1, [%[outptr5], #1, MUL VL]\n"
                            "fmin z13.h, p1/m, z13.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[outptr3], #0x60]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "st1h z11.h, p2, [%[outptr5], #2, MUL VL]\n"
                            "fmax z12.h, p0/m, z12.h, z1.h\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                            "fmax z13.h, p1/m, z13.h, z1.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x240]\n"
                            "fmin z14.h, p2/m, z14.h, z0.h\n"
                            "st1h z12.h, p0, [%[outptr6]]\n"
                            "prfm PLDL1KEEP, [%[outptr4], #0x60]\n"
                            "addvl %[outptr4], %[outptr4], #3\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x280]\n"
                            "fmax z14.h, p2/m, z14.h, z1.h\n"
                            "st1h z13.h, p1, [%[outptr6], #1, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr5], #0x60]\n"
                            "addvl %[outptr5], %[outptr5], #3\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x2c0]\n"
                            "st1h z14.h, p2, [%[outptr6], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr6], #0x60]\n"
                            "addvl %[outptr6], %[outptr6], #3\n"
                            "addvl %[inptr], %[inptr], #24\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                default:
                case 8:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PLDL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[outptr0]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z10.h, p0/z, [%[inptr]]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z5.h, p0/z, [%[outptr1]]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "ld1h z3.h, p1/z, [%[outptr0], #1, MUL VL]\n"
                            "ld1h z11.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z13.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[outptr1], #0x60]\n"
                            "fmin z10.h, p0/m, z10.h, z0.h\n"
                            "ld1h z4.h, p2/z, [%[outptr0], #2, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z12.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z6.h, p1/z, [%[outptr1], #1, MUL VL]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fmax z10.h, p0/m, z10.h, z1.h\n"
                            "ld1h z7.h, p2/z, [%[outptr1], #2, MUL VL]\n"
                            "fmin z11.h, p1/m, z11.h, z0.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z8.h, p0/z, [%[outptr2]]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "st1h z10.h, p0, [%[outptr0]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "ld1h z16.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fmax z11.h, p1/m, z11.h, z1.h\n"
                            "ld1h z9.h, p1/z, [%[outptr2], #1, MUL VL]\n"
                            "fmin z12.h, p2/m, z12.h, z0.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "ld1h z2.h, p2/z, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "st1h z11.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z10.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fmax z12.h, p2/m, z12.h, z1.h\n"
                            "ld1h z3.h, p0/z, [%[outptr3]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "ld1h z11.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "ld1h z4.h, p1/z, [%[outptr3], #1, MUL VL]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z12.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "ld1h z12.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z5.h, p2/z, [%[outptr3], #2, MUL VL]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z13.h, p0, [%[outptr1]]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "ld1h z13.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "ld1h z6.h, p0/z, [%[outptr4]]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmin z10.h, p2/m, z10.h, z0.h\n"
                            "ld1h z14.h, p0/z, [x8, #-4, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z7.h, p1/z, [%[outptr4], #1, MUL VL]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z15.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z15.h, p1/z, [x8, #-3, MUL VL]\n"
                            "fmax z10.h, p2/m, z10.h, z1.h\n"
                            "ld1h z8.h, p2/z, [%[outptr4], #2, MUL VL]\n"
                            "fmin z11.h, p0/m, z11.h, z0.h\n"
                            "st1h z16.h, p0, [%[outptr2]]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z16.h, p2/z, [x8, #-2, MUL VL]\n"
                            "fmin z12.h, p1/m, z12.h, z0.h\n"
                            "ld1h z9.h, p0/z, [%[outptr5]]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "st1h z17.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fmax z11.h, p0/m, z11.h, z1.h\n"
                            "ld1h z17.h, p0/z, [x8, #-1, MUL VL]\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "ld1h z2.h, p1/z, [%[outptr5], #1, MUL VL]\n"
                            "fmax z12.h, p1/m, z12.h, z1.h\n"
                            "st1h z10.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "ld1h z10.h, p1/z, [x8]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "ld1h z3.h, p2/z, [%[outptr5], #2, MUL VL]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "st1h z11.h, p0, [%[outptr3]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "ld1h z11.h, p2/z, [x8, #1, MUL VL]\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "ld1h z4.h, p0/z, [%[outptr6]]\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "st1h z12.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "ld1h z12.h, p0/z, [x8, #2, MUL VL]\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "ld1h z5.h, p1/z, [%[outptr6], #1, MUL VL]\n"
                            "fadd z10.h, z10.h, z2.h\n"
                            "st1h z13.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "ld1h z13.h, p1/z, [x8, #3, MUL VL]\n"
                            "fmin z17.h, p0/m, z17.h, z0.h\n"
                            "ld1h z6.h, p2/z, [%[outptr6], #2, MUL VL]\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "st1h z14.h, p0, [%[outptr4]]\n"
                            "fmin z10.h, p1/m, z10.h, z0.h\n"
                            "ld1h z14.h, p2/z, [x8, #4, MUL VL]\n"
                            "fadd z11.h, z11.h, z3.h\n"
                            "ld1h z7.h, p0/z, [%[outptr7]]\n"
                            "fmax z17.h, p0/m, z17.h, z1.h\n"
                            "st1h z15.h, p1, [%[outptr4], #1, MUL VL]\n"
                            "fadd z12.h, z12.h, z4.h\n"
                            "ld1h z15.h, p0/z, [x8, #5, MUL VL]\n"
                            "fmax z10.h, p1/m, z10.h, z1.h\n"
                            "ld1h z8.h, p1/z, [%[outptr7], #1, MUL VL]\n"
                            "fmin z11.h, p2/m, z11.h, z0.h\n"
                            "st1h z16.h, p2, [%[outptr4], #2, MUL VL]\n"
                            "fadd z13.h, z13.h, z5.h\n"
                            "ld1h z16.h, p1/z, [x8, #6, MUL VL]\n"
                            "fmin z12.h, p0/m, z12.h, z0.h\n"
                            "ld1h z9.h, p2/z, [%[outptr7], #2, MUL VL]\n"
                            "fadd z14.h, z14.h, z6.h\n"
                            "st1h z17.h, p0, [%[outptr5]]\n"
                            "fmax z11.h, p2/m, z11.h, z1.h\n"
                            "ld1h z17.h, p2/z, [x8, #7, MUL VL]\n"
                            "fmin z13.h, p1/m, z13.h, z0.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmax z12.h, p0/m, z12.h, z1.h\n"
                            "st1h z10.h, p1, [%[outptr5], #1, MUL VL]\n"
                            "fmin z14.h, p2/m, z14.h, z0.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmax z13.h, p1/m, z13.h, z1.h\n"
                            "st1h z11.h, p2, [%[outptr5], #2, MUL VL]\n"
                            "fadd z15.h, z15.h, z7.h\n"
                            "prfm PLDL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z14.h, p2/m, z14.h, z1.h\n"
                            "st1h z12.h, p0, [%[outptr6]]\n"
                            "fadd z16.h, z16.h, z8.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmin z15.h, p0/m, z15.h, z0.h\n"
                            "st1h z13.h, p1, [%[outptr6], #1, MUL VL]\n"
                            "fadd z17.h, z17.h, z9.h\n"
                            "prfm PLDL1KEEP, [%[outptr3], #0x60]\n"
                            "fmin z16.h, p1/m, z16.h, z0.h\n"
                            "st1h z14.h, p2, [%[outptr6], #2, MUL VL]\n"
                            "fmax z15.h, p0/m, z15.h, z1.h\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                            "fmin z17.h, p2/m, z17.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x240]\n"
                            "fmax z16.h, p1/m, z16.h, z1.h\n"
                            "st1h z15.h, p0, [%[outptr7]]\n"
                            "prfm PLDL1KEEP, [%[outptr4], #0x60]\n"
                            "fmax z17.h, p2/m, z17.h, z1.h\n"
                            "addvl %[outptr4], %[outptr4], #3\n"
                            "st1h z16.h, p1, [%[outptr7], #1, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x280]\n"
                            "prfm PLDL1KEEP, [%[outptr5], #0x60]\n"
                            "addvl %[outptr5], %[outptr5], #3\n"
                            "st1h z17.h, p2, [%[outptr7], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x2c0]\n"
                            "prfm PLDL1KEEP, [%[outptr6], #0x60]\n"
                            "addvl %[outptr6], %[outptr6], #3\n"
                            "prfm PLDL1KEEP, [%[outptr7], #0x60]\n"
                            "addvl %[outptr7], %[outptr7], #3\n"
                            "addvl %[inptr], %[inptr], #24\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;


                }
            }
            else
            {
                const __fp16 *biasptr = bias ? bias + i : nullbias;

                switch(height)
                {
                case 1:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PSTL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[biasptr]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z3.h, p0/z, [%[biasptr], #1, MUL VL]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z4.h, p0/z, [%[biasptr], #2, MUL VL]\n"
                            "ld1h z13.h, p0/z, [%[inptr]]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "fadd z13.h, z13.h, z2.h\n"
                            "fadd z14.h, z14.h, z3.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "fadd z15.h, z15.h, z4.h\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z13.h, p0, [%[outptr0]]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "st1h z15.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [biasptr] "r" (biasptr), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 2:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PSTL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[biasptr]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z3.h, p0/z, [%[biasptr], #1, MUL VL]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z4.h, p0/z, [%[biasptr], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "ld1h z13.h, p0/z, [%[inptr]]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "prfm PSTL1KEEP, [%[outptr1], #0x60]\n"
                            "fadd z13.h, z13.h, z2.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "ld1h z16.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "fadd z14.h, z14.h, z3.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "ld1h z18.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "fadd z15.h, z15.h, z4.h\n"
                            "fadd z16.h, z16.h, z2.h\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "st1h z13.h, p0, [%[outptr0]]\n"
                            "fadd z17.h, z17.h, z3.h\n"
                            "fadd z18.h, z18.h, z4.h\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "fmin z18.h, p2/m, z18.h, z0.h\n"
                            "st1h z15.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z16.h, p0, [%[outptr1]]\n"
                            "fmax z18.h, p2/m, z18.h, z1.h\n"
                            "st1h z17.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "st1h z18.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [biasptr] "r" (biasptr), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 3:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PSTL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[biasptr]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z3.h, p0/z, [%[biasptr], #1, MUL VL]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z4.h, p0/z, [%[biasptr], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "ld1h z13.h, p0/z, [%[inptr]]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "prfm PSTL1KEEP, [%[outptr1], #0x60]\n"
                            "fadd z13.h, z13.h, z2.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "ld1h z16.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fadd z14.h, z14.h, z3.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "ld1h z18.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z15.h, z15.h, z4.h\n"
                            "ld1h z19.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fadd z16.h, z16.h, z2.h\n"
                            "ld1h z20.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "prfm PSTL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z13.h, p0, [%[outptr0]]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z13.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fadd z17.h, z17.h, z3.h\n"
                            "fadd z18.h, z18.h, z4.h\n"
                            "st1h z14.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "fmin z18.h, p2/m, z18.h, z0.h\n"
                            "st1h z15.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z19.h, z19.h, z2.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z16.h, p0, [%[outptr1]]\n"
                            "fmax z18.h, p2/m, z18.h, z1.h\n"
                            "fmin z19.h, p0/m, z19.h, z0.h\n"
                            "fadd z20.h, z20.h, z3.h\n"
                            "st1h z17.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fadd z13.h, z13.h, z4.h\n"
                            "fmax z19.h, p0/m, z19.h, z1.h\n"
                            "st1h z18.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fmin z20.h, p1/m, z20.h, z0.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "st1h z19.h, p0, [%[outptr2]]\n"
                            "fmax z20.h, p1/m, z20.h, z1.h\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "st1h z20.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "st1h z13.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [biasptr] "r" (biasptr), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 4:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PSTL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[biasptr]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z3.h, p0/z, [%[biasptr], #1, MUL VL]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z4.h, p0/z, [%[biasptr], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "ld1h z13.h, p0/z, [%[inptr]]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "prfm PSTL1KEEP, [%[outptr1], #0x60]\n"
                            "fadd z13.h, z13.h, z2.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "ld1h z16.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fadd z14.h, z14.h, z3.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "ld1h z18.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z15.h, z15.h, z4.h\n"
                            "ld1h z19.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fadd z16.h, z16.h, z2.h\n"
                            "ld1h z20.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "prfm PSTL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "prfm PSTL1KEEP, [%[outptr3], #0x60]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z13.h, p0, [%[outptr0]]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z13.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fadd z17.h, z17.h, z3.h\n"
                            "fadd z18.h, z18.h, z4.h\n"
                            "st1h z14.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "ld1h z14.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "fmin z18.h, p2/m, z18.h, z0.h\n"
                            "st1h z15.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z19.h, z19.h, z2.h\n"
                            "ld1h z15.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fadd z20.h, z20.h, z3.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "st1h z16.h, p0, [%[outptr1]]\n"
                            "fmax z18.h, p2/m, z18.h, z1.h\n"
                            "ld1h z16.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fmin z19.h, p0/m, z19.h, z0.h\n"
                            "fmin z20.h, p1/m, z20.h, z0.h\n"
                            "st1h z17.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fadd z13.h, z13.h, z4.h\n"
                            "fadd z14.h, z14.h, z2.h\n"
                            "fmax z19.h, p0/m, z19.h, z1.h\n"
                            "st1h z18.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fmax z20.h, p1/m, z20.h, z1.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "st1h z19.h, p0, [%[outptr2]]\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "fadd z15.h, z15.h, z3.h\n"
                            "fadd z16.h, z16.h, z4.h\n"
                            "st1h z20.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "st1h z13.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "st1h z14.h, p0, [%[outptr3]]\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "st1h z15.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "st1h z16.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [biasptr] "r" (biasptr), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 5:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PSTL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[biasptr]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z3.h, p0/z, [%[biasptr], #1, MUL VL]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z4.h, p0/z, [%[biasptr], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "ld1h z13.h, p0/z, [%[inptr]]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "prfm PSTL1KEEP, [%[outptr1], #0x60]\n"
                            "fadd z13.h, z13.h, z2.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "ld1h z16.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fadd z14.h, z14.h, z3.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "ld1h z18.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z15.h, z15.h, z4.h\n"
                            "ld1h z19.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fadd z16.h, z16.h, z2.h\n"
                            "ld1h z20.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "prfm PSTL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "prfm PSTL1KEEP, [%[outptr3], #0x60]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x240]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z13.h, p0, [%[outptr0]]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z13.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fadd z17.h, z17.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr4], #0x60]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "ld1h z14.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fadd z18.h, z18.h, z4.h\n"
                            "st1h z15.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z19.h, z19.h, z2.h\n"
                            "ld1h z15.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmin z18.h, p2/m, z18.h, z0.h\n"
                            "st1h z16.h, p0, [%[outptr1]]\n"
                            "fmin z19.h, p0/m, z19.h, z0.h\n"
                            "ld1h z16.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fadd z20.h, z20.h, z3.h\n"
                            "fadd z13.h, z13.h, z4.h\n"
                            "st1h z17.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmax z18.h, p2/m, z18.h, z1.h\n"
                            "ld1h z17.h, p0/z, [x8, #-4, MUL VL]\n"
                            "fmax z19.h, p0/m, z19.h, z1.h\n"
                            "fmin z20.h, p1/m, z20.h, z0.h\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "st1h z18.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z14.h, z14.h, z2.h\n"
                            "ld1h z18.h, p1/z, [x8, #-3, MUL VL]\n"
                            "fadd z15.h, z15.h, z3.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmax z20.h, p1/m, z20.h, z1.h\n"
                            "st1h z19.h, p0, [%[outptr2]]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "ld1h z19.h, p2/z, [x8, #-2, MUL VL]\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "st1h z20.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fadd z16.h, z16.h, z4.h\n"
                            "fadd z17.h, z17.h, z2.h\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "st1h z13.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "st1h z14.h, p0, [%[outptr3]]\n"
                            "fmin z17.h, p0/m, z17.h, z0.h\n"
                            "fadd z18.h, z18.h, z3.h\n"
                            "fadd z19.h, z19.h, z4.h\n"
                            "st1h z15.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "fmax z17.h, p0/m, z17.h, z1.h\n"
                            "fmin z18.h, p1/m, z18.h, z0.h\n"
                            "fmin z19.h, p2/m, z19.h, z0.h\n"
                            "st1h z16.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                            "fmax z18.h, p1/m, z18.h, z1.h\n"
                            "st1h z17.h, p0, [%[outptr4]]\n"
                            "fmax z19.h, p2/m, z19.h, z1.h\n"
                            "st1h z18.h, p1, [%[outptr4], #1, MUL VL]\n"
                            "st1h z19.h, p2, [%[outptr4], #2, MUL VL]\n"
                            "addvl %[outptr4], %[outptr4], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [biasptr] "r" (biasptr), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 6:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PSTL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[biasptr]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z3.h, p0/z, [%[biasptr], #1, MUL VL]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z4.h, p0/z, [%[biasptr], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "ld1h z13.h, p0/z, [%[inptr]]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "prfm PSTL1KEEP, [%[outptr1], #0x60]\n"
                            "fadd z13.h, z13.h, z2.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "ld1h z16.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fadd z14.h, z14.h, z3.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "ld1h z18.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z15.h, z15.h, z4.h\n"
                            "ld1h z19.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fadd z16.h, z16.h, z2.h\n"
                            "ld1h z20.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "prfm PSTL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "prfm PSTL1KEEP, [%[outptr3], #0x60]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x240]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z13.h, p0, [%[outptr0]]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z13.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fadd z17.h, z17.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr4], #0x60]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "ld1h z14.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x280]\n"
                            "fadd z18.h, z18.h, z4.h\n"
                            "st1h z15.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z19.h, z19.h, z2.h\n"
                            "ld1h z15.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmin z18.h, p2/m, z18.h, z0.h\n"
                            "st1h z16.h, p0, [%[outptr1]]\n"
                            "fmin z19.h, p0/m, z19.h, z0.h\n"
                            "ld1h z16.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fadd z20.h, z20.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr5], #0x60]\n"
                            "fmax z18.h, p2/m, z18.h, z1.h\n"
                            "st1h z17.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmax z19.h, p0/m, z19.h, z1.h\n"
                            "ld1h z17.h, p0/z, [x8, #-4, MUL VL]\n"
                            "fmin z20.h, p1/m, z20.h, z0.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fadd z13.h, z13.h, z4.h\n"
                            "st1h z18.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z14.h, z14.h, z2.h\n"
                            "ld1h z18.h, p1/z, [x8, #-3, MUL VL]\n"
                            "fmax z20.h, p1/m, z20.h, z1.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "st1h z19.h, p0, [%[outptr2]]\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "ld1h z19.h, p2/z, [x8, #-2, MUL VL]\n"
                            "fadd z15.h, z15.h, z3.h\n"
                            "fadd z16.h, z16.h, z4.h\n"
                            "st1h z20.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "ld1h z20.h, p0/z, [x8, #-1, MUL VL]\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "st1h z13.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fadd z17.h, z17.h, z2.h\n"
                            "ld1h z13.h, p1/z, [x8]\n"
                            "fadd z18.h, z18.h, z3.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "st1h z14.h, p0, [%[outptr3]]\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "ld1h z14.h, p2/z, [x8, #1, MUL VL]\n"
                            "fmin z17.h, p0/m, z17.h, z0.h\n"
                            "fmin z18.h, p1/m, z18.h, z0.h\n"
                            "st1h z15.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "fadd z19.h, z19.h, z4.h\n"
                            "fadd z20.h, z20.h, z2.h\n"
                            "fmax z17.h, p0/m, z17.h, z1.h\n"
                            "st1h z16.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "fmax z18.h, p1/m, z18.h, z1.h\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                            "fmin z19.h, p2/m, z19.h, z0.h\n"
                            "st1h z17.h, p0, [%[outptr4]]\n"
                            "fmin z20.h, p0/m, z20.h, z0.h\n"
                            "fadd z13.h, z13.h, z3.h\n"
                            "fadd z14.h, z14.h, z4.h\n"
                            "st1h z18.h, p1, [%[outptr4], #1, MUL VL]\n"
                            "fmax z19.h, p2/m, z19.h, z1.h\n"
                            "fmax z20.h, p0/m, z20.h, z1.h\n"
                            "fmin z13.h, p1/m, z13.h, z0.h\n"
                            "fmin z14.h, p2/m, z14.h, z0.h\n"
                            "st1h z19.h, p2, [%[outptr4], #2, MUL VL]\n"
                            "addvl %[outptr4], %[outptr4], #3\n"
                            "fmax z13.h, p1/m, z13.h, z1.h\n"
                            "st1h z20.h, p0, [%[outptr5]]\n"
                            "fmax z14.h, p2/m, z14.h, z1.h\n"
                            "st1h z13.h, p1, [%[outptr5], #1, MUL VL]\n"
                            "st1h z14.h, p2, [%[outptr5], #2, MUL VL]\n"
                            "addvl %[outptr5], %[outptr5], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [biasptr] "r" (biasptr), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                case 7:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PSTL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[biasptr]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z3.h, p0/z, [%[biasptr], #1, MUL VL]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z4.h, p0/z, [%[biasptr], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "ld1h z13.h, p0/z, [%[inptr]]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "prfm PSTL1KEEP, [%[outptr1], #0x60]\n"
                            "fadd z13.h, z13.h, z2.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "ld1h z16.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fadd z14.h, z14.h, z3.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "ld1h z18.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z15.h, z15.h, z4.h\n"
                            "ld1h z19.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fadd z16.h, z16.h, z2.h\n"
                            "ld1h z20.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "prfm PSTL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "prfm PSTL1KEEP, [%[outptr3], #0x60]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x240]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z13.h, p0, [%[outptr0]]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z13.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fadd z17.h, z17.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr4], #0x60]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "ld1h z14.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x280]\n"
                            "fadd z18.h, z18.h, z4.h\n"
                            "st1h z15.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z19.h, z19.h, z2.h\n"
                            "ld1h z15.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmin z18.h, p2/m, z18.h, z0.h\n"
                            "st1h z16.h, p0, [%[outptr1]]\n"
                            "fmin z19.h, p0/m, z19.h, z0.h\n"
                            "ld1h z16.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fadd z20.h, z20.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr5], #0x60]\n"
                            "fmax z18.h, p2/m, z18.h, z1.h\n"
                            "st1h z17.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmax z19.h, p0/m, z19.h, z1.h\n"
                            "ld1h z17.h, p0/z, [x8, #-4, MUL VL]\n"
                            "fmin z20.h, p1/m, z20.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x2c0]\n"
                            "fadd z13.h, z13.h, z4.h\n"
                            "st1h z18.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z14.h, z14.h, z2.h\n"
                            "ld1h z18.h, p1/z, [x8, #-3, MUL VL]\n"
                            "fmax z20.h, p1/m, z20.h, z1.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "st1h z19.h, p0, [%[outptr2]]\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "ld1h z19.h, p2/z, [x8, #-2, MUL VL]\n"
                            "fadd z15.h, z15.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr6], #0x60]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "st1h z20.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "ld1h z20.h, p0/z, [x8, #-1, MUL VL]\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fadd z16.h, z16.h, z4.h\n"
                            "st1h z13.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fadd z17.h, z17.h, z2.h\n"
                            "ld1h z13.h, p1/z, [x8]\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "st1h z14.h, p0, [%[outptr3]]\n"
                            "fmin z17.h, p0/m, z17.h, z0.h\n"
                            "ld1h z14.h, p2/z, [x8, #1, MUL VL]\n"
                            "fadd z18.h, z18.h, z3.h\n"
                            "fadd z19.h, z19.h, z4.h\n"
                            "st1h z15.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "ld1h z15.h, p0/z, [x8, #2, MUL VL]\n"
                            "fmax z17.h, p0/m, z17.h, z1.h\n"
                            "fmin z18.h, p1/m, z18.h, z0.h\n"
                            "fmin z19.h, p2/m, z19.h, z0.h\n"
                            "st1h z16.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "fadd z20.h, z20.h, z2.h\n"
                            "ld1h z16.h, p1/z, [x8, #3, MUL VL]\n"
                            "fadd z13.h, z13.h, z3.h\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                            "fmax z18.h, p1/m, z18.h, z1.h\n"
                            "st1h z17.h, p0, [%[outptr4]]\n"
                            "fmax z19.h, p2/m, z19.h, z1.h\n"
                            "ld1h z17.h, p2/z, [x8, #4, MUL VL]\n"
                            "fmin z20.h, p0/m, z20.h, z0.h\n"
                            "fmin z13.h, p1/m, z13.h, z0.h\n"
                            "st1h z18.h, p1, [%[outptr4], #1, MUL VL]\n"
                            "fadd z14.h, z14.h, z4.h\n"
                            "fadd z15.h, z15.h, z2.h\n"
                            "fmax z20.h, p0/m, z20.h, z1.h\n"
                            "st1h z19.h, p2, [%[outptr4], #2, MUL VL]\n"
                            "fmax z13.h, p1/m, z13.h, z1.h\n"
                            "addvl %[outptr4], %[outptr4], #3\n"
                            "fmin z14.h, p2/m, z14.h, z0.h\n"
                            "st1h z20.h, p0, [%[outptr5]]\n"
                            "fmin z15.h, p0/m, z15.h, z0.h\n"
                            "fadd z16.h, z16.h, z3.h\n"
                            "fadd z17.h, z17.h, z4.h\n"
                            "st1h z13.h, p1, [%[outptr5], #1, MUL VL]\n"
                            "fmax z14.h, p2/m, z14.h, z1.h\n"
                            "fmax z15.h, p0/m, z15.h, z1.h\n"
                            "fmin z16.h, p1/m, z16.h, z0.h\n"
                            "fmin z17.h, p2/m, z17.h, z0.h\n"
                            "st1h z14.h, p2, [%[outptr5], #2, MUL VL]\n"
                            "addvl %[outptr5], %[outptr5], #3\n"
                            "fmax z16.h, p1/m, z16.h, z1.h\n"
                            "st1h z15.h, p0, [%[outptr6]]\n"
                            "fmax z17.h, p2/m, z17.h, z1.h\n"
                            "st1h z16.h, p1, [%[outptr6], #1, MUL VL]\n"
                            "st1h z17.h, p2, [%[outptr6], #2, MUL VL]\n"
                            "addvl %[outptr6], %[outptr6], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [biasptr] "r" (biasptr), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;

                default:
                case 8:
                    {
                        long w = xmax - i;
                        long p = 0;
                        /* Optimized routine to copy an entire block */
                        __asm __volatile (
                            "mov z0.h, %h[maxval]\n"
                            "addvl x8, %[inptr], #16\n"
                            "mov z1.h, %h[minval]\n"
                            "whilelt p0.h, %[p], %[w]\n"
                            "inch %[p], all, mul #1\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x180]\n"
                            "prfm PSTL1KEEP, [%[outptr0], #0x60]\n"
                            "ld1h z2.h, p0/z, [%[biasptr]]\n"
                            "whilelt p1.h, %[p], %[w]\n"
                            "ld1h z3.h, p0/z, [%[biasptr], #1, MUL VL]\n"
                            "inch %[p], all, mul #1\n"
                            "ld1h z4.h, p0/z, [%[biasptr], #2, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x1c0]\n"
                            "ld1h z13.h, p0/z, [%[inptr]]\n"
                            "whilelt p2.h, %[p], %[w]\n"
                            "ld1h z14.h, p1/z, [%[inptr], #1, MUL VL]\n"
                            "prfm PSTL1KEEP, [%[outptr1], #0x60]\n"
                            "fadd z13.h, z13.h, z2.h\n"
                            "ld1h z15.h, p2/z, [%[inptr], #2, MUL VL]\n"
                            "ld1h z16.h, p0/z, [%[inptr], #3, MUL VL]\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x200]\n"
                            "fadd z14.h, z14.h, z3.h\n"
                            "ld1h z17.h, p1/z, [%[inptr], #4, MUL VL]\n"
                            "fmin z13.h, p0/m, z13.h, z0.h\n"
                            "ld1h z18.h, p2/z, [%[inptr], #5, MUL VL]\n"
                            "fadd z15.h, z15.h, z4.h\n"
                            "ld1h z19.h, p0/z, [%[inptr], #6, MUL VL]\n"
                            "fadd z16.h, z16.h, z2.h\n"
                            "ld1h z20.h, p1/z, [%[inptr], #7, MUL VL]\n"
                            "fmin z14.h, p1/m, z14.h, z0.h\n"
                            "prfm PSTL1KEEP, [%[outptr2], #0x60]\n"
                            "fmax z13.h, p0/m, z13.h, z1.h\n"
                            "prfm PSTL1KEEP, [%[outptr3], #0x60]\n"
                            "fmax z14.h, p1/m, z14.h, z1.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x240]\n"
                            "fmin z15.h, p2/m, z15.h, z0.h\n"
                            "st1h z13.h, p0, [%[outptr0]]\n"
                            "fmin z16.h, p0/m, z16.h, z0.h\n"
                            "ld1h z13.h, p2/z, [x8, #-8, MUL VL]\n"
                            "fadd z17.h, z17.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr4], #0x60]\n"
                            "fmax z15.h, p2/m, z15.h, z1.h\n"
                            "st1h z14.h, p1, [%[outptr0], #1, MUL VL]\n"
                            "fmax z16.h, p0/m, z16.h, z1.h\n"
                            "ld1h z14.h, p0/z, [x8, #-7, MUL VL]\n"
                            "fmin z17.h, p1/m, z17.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x280]\n"
                            "fadd z18.h, z18.h, z4.h\n"
                            "st1h z15.h, p2, [%[outptr0], #2, MUL VL]\n"
                            "fadd z19.h, z19.h, z2.h\n"
                            "ld1h z15.h, p1/z, [x8, #-6, MUL VL]\n"
                            "fmax z17.h, p1/m, z17.h, z1.h\n"
                            "addvl %[outptr0], %[outptr0], #3\n"
                            "fmin z18.h, p2/m, z18.h, z0.h\n"
                            "st1h z16.h, p0, [%[outptr1]]\n"
                            "fmin z19.h, p0/m, z19.h, z0.h\n"
                            "ld1h z16.h, p2/z, [x8, #-5, MUL VL]\n"
                            "fadd z20.h, z20.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr5], #0x60]\n"
                            "fmax z18.h, p2/m, z18.h, z1.h\n"
                            "st1h z17.h, p1, [%[outptr1], #1, MUL VL]\n"
                            "fmax z19.h, p0/m, z19.h, z1.h\n"
                            "ld1h z17.h, p0/z, [x8, #-4, MUL VL]\n"
                            "fmin z20.h, p1/m, z20.h, z0.h\n"
                            "prfm PLDL1KEEP, [%[inptr], #0x2c0]\n"
                            "fadd z13.h, z13.h, z4.h\n"
                            "st1h z18.h, p2, [%[outptr1], #2, MUL VL]\n"
                            "fadd z14.h, z14.h, z2.h\n"
                            "ld1h z18.h, p1/z, [x8, #-3, MUL VL]\n"
                            "fmax z20.h, p1/m, z20.h, z1.h\n"
                            "addvl %[outptr1], %[outptr1], #3\n"
                            "fmin z13.h, p2/m, z13.h, z0.h\n"
                            "st1h z19.h, p0, [%[outptr2]]\n"
                            "fmin z14.h, p0/m, z14.h, z0.h\n"
                            "ld1h z19.h, p2/z, [x8, #-2, MUL VL]\n"
                            "fadd z15.h, z15.h, z3.h\n"
                            "prfm PSTL1KEEP, [%[outptr6], #0x60]\n"
                            "fmax z13.h, p2/m, z13.h, z1.h\n"
                            "st1h z20.h, p1, [%[outptr2], #1, MUL VL]\n"
                            "fmax z14.h, p0/m, z14.h, z1.h\n"
                            "ld1h z20.h, p0/z, [x8, #-1, MUL VL]\n"
                            "fmin z15.h, p1/m, z15.h, z0.h\n"
                            "prfm PSTL1KEEP, [%[outptr7], #0x60]\n"
                            "fadd z16.h, z16.h, z4.h\n"
                            "st1h z13.h, p2, [%[outptr2], #2, MUL VL]\n"
                            "fadd z17.h, z17.h, z2.h\n"
                            "ld1h z13.h, p1/z, [x8]\n"
                            "fmax z15.h, p1/m, z15.h, z1.h\n"
                            "addvl %[outptr2], %[outptr2], #3\n"
                            "fmin z16.h, p2/m, z16.h, z0.h\n"
                            "st1h z14.h, p0, [%[outptr3]]\n"
                            "fmin z17.h, p0/m, z17.h, z0.h\n"
                            "ld1h z14.h, p2/z, [x8, #1, MUL VL]\n"
                            "fadd z18.h, z18.h, z3.h\n"
                            "addvl %[inptr], %[inptr], #24\n"
                            "fmax z16.h, p2/m, z16.h, z1.h\n"
                            "st1h z15.h, p1, [%[outptr3], #1, MUL VL]\n"
                            "fmax z17.h, p0/m, z17.h, z1.h\n"
                            "ld1h z15.h, p0/z, [x8, #2, MUL VL]\n"
                            "fmin z18.h, p1/m, z18.h, z0.h\n"
                            "fadd z19.h, z19.h, z4.h\n"
                            "st1h z16.h, p2, [%[outptr3], #2, MUL VL]\n"
                            "fadd z20.h, z20.h, z2.h\n"
                            "ld1h z16.h, p1/z, [x8, #3, MUL VL]\n"
                            "fadd z13.h, z13.h, z3.h\n"
                            "addvl %[outptr3], %[outptr3], #3\n"
                            "fmax z18.h, p1/m, z18.h, z1.h\n"
                            "st1h z17.h, p0, [%[outptr4]]\n"
                            "fmin z19.h, p2/m, z19.h, z0.h\n"
                            "ld1h z17.h, p2/z, [x8, #4, MUL VL]\n"
                            "fmin z20.h, p0/m, z20.h, z0.h\n"
                            "fmin z13.h, p1/m, z13.h, z0.h\n"
                            "st1h z18.h, p1, [%[outptr4], #1, MUL VL]\n"
                            "fadd z14.h, z14.h, z4.h\n"
                            "ld1h z18.h, p0/z, [x8, #5, MUL VL]\n"
                            "fmax z19.h, p2/m, z19.h, z1.h\n"
                            "fmax z20.h, p0/m, z20.h, z1.h\n"
                            "fmax z13.h, p1/m, z13.h, z1.h\n"
                            "fmin z14.h, p2/m, z14.h, z0.h\n"
                            "st1h z19.h, p2, [%[outptr4], #2, MUL VL]\n"
                            "fadd z15.h, z15.h, z2.h\n"
                            "ld1h z19.h, p1/z, [x8, #6, MUL VL]\n"
                            "fadd z16.h, z16.h, z3.h\n"
                            "addvl %[outptr4], %[outptr4], #3\n"
                            "fmax z14.h, p2/m, z14.h, z1.h\n"
                            "st1h z20.h, p0, [%[outptr5]]\n"
                            "fmin z15.h, p0/m, z15.h, z0.h\n"
                            "ld1h z20.h, p2/z, [x8, #7, MUL VL]\n"
                            "fmin z16.h, p1/m, z16.h, z0.h\n"
                            "fadd z17.h, z17.h, z4.h\n"
                            "st1h z13.h, p1, [%[outptr5], #1, MUL VL]\n"
                            "fadd z18.h, z18.h, z2.h\n"
                            "fmax z15.h, p0/m, z15.h, z1.h\n"
                            "fmax z16.h, p1/m, z16.h, z1.h\n"
                            "st1h z14.h, p2, [%[outptr5], #2, MUL VL]\n"
                            "fmin z17.h, p2/m, z17.h, z0.h\n"
                            "addvl %[outptr5], %[outptr5], #3\n"
                            "fmin z18.h, p0/m, z18.h, z0.h\n"
                            "st1h z15.h, p0, [%[outptr6]]\n"
                            "fadd z19.h, z19.h, z3.h\n"
                            "fmax z17.h, p2/m, z17.h, z1.h\n"
                            "fadd z20.h, z20.h, z4.h\n"
                            "st1h z16.h, p1, [%[outptr6], #1, MUL VL]\n"
                            "fmax z18.h, p0/m, z18.h, z1.h\n"
                            "fmin z19.h, p1/m, z19.h, z0.h\n"
                            "fmin z20.h, p2/m, z20.h, z0.h\n"
                            "st1h z17.h, p2, [%[outptr6], #2, MUL VL]\n"
                            "addvl %[outptr6], %[outptr6], #3\n"
                            "fmax z19.h, p1/m, z19.h, z1.h\n"
                            "fmax z20.h, p2/m, z20.h, z1.h\n"
                            "st1h z18.h, p0, [%[outptr7]]\n"
                            "st1h z19.h, p1, [%[outptr7], #1, MUL VL]\n"
                            "st1h z20.h, p2, [%[outptr7], #2, MUL VL]\n"
                            "addvl %[outptr7], %[outptr7], #3\n"
                        : [outptr0] "+r" (outptr0), [outptr1] "+r" (outptr1), [outptr2] "+r" (outptr2), [outptr3] "+r" (outptr3), [outptr4] "+r" (outptr4), [outptr5] "+r" (outptr5), [outptr6] "+r" (outptr6), [outptr7] "+r" (outptr7),
                          [inptr] "+r" (inptr), [p] "+r" (p)
                        : [w] "r" (w), [biasptr] "r" (biasptr), [minval] "w" (minval), [maxval] "w" (maxval)
                        : "x8", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "memory", "cc"
                        );
                    }
                    break;


                }
            }
        }
    }
}

#endif // __ARM_FEATURE_SVE
