// @HEADER
// ************************************************************************
//
//                           Intrepid2 Package
//                 Copyright (2007) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact Kyungjoo Kim  (kyukim@sandia.gov), or
//                    Mauro Perego  (mperego@sandia.gov)
//
// ************************************************************************
// @HEADER

/** \file test_01.hpp
 \brief  Unit tests for the Intrepid2::HGRAD_TET_CN_FEM_ORTH class.
 \author Created by R. Kirby, M. Perego and K. Kim.
 */

#include "Intrepid2_config.h"

#ifdef HAVE_INTREPID2_DEBUG
#define INTREPID2_TEST_FOR_DEBUG_ABORT_OVERRIDE_TO_CONTINUE
#endif

#include "Intrepid2_HGRAD_TET_Cn_FEM_ORTH.hpp"
#include "Intrepid2_PointTools.hpp"


#include "Teuchos_oblackholestream.hpp"
#include "Teuchos_RCP.hpp"
#include "Intrepid2_CubatureDirectTetDefault.hpp"

namespace Intrepid2 {

namespace Test {
#define INTREPID2_TEST_ERROR_EXPECTED( S )                              \
    try {                                                               \
      ++nthrow;                                                         \
      S ;                                                               \
    }                                                                   \
    catch (std::exception &err) {                                        \
      ++ncatch;                                                         \
      *outStream << "Expected Error ----------------------------------------------------------------\n"; \
      *outStream << err.what() << '\n';                                 \
      *outStream << "-------------------------------------------------------------------------------" << "\n\n"; \
    }

template<typename ValueType, typename DeviceType>
int HGRAD_TET_Cn_FEM_ORTH_Test01(const bool verbose) {

  Teuchos::RCP<std::ostream> outStream;
  Teuchos::oblackholestream bhs; // outputs nothing

  if (verbose)
    outStream = Teuchos::rcp(&std::cout, false);
  else
    outStream = Teuchos::rcp(&bhs, false);

  Teuchos::oblackholestream oldFormatState;
  oldFormatState.copyfmt(std::cout);

  *outStream
  << "===============================================================================\n"
  << "|                                                                             |\n"
  << "|                           Unit Test OrthogonalBases                         |\n"
  << "|                                                                             |\n"
  << "|     1) Tests orthogonality of tetrahedral orthogonal basis                   |\n"
  << "|                                                                             |\n"
  << "|  Questions? Contact  Pavel Bochev (pbboche@sandia.gov),                     |\n"
  << "|                      Denis Ridzal (dridzal@sandia.gov),                     |\n"
  << "|                      Robert Kirby (robert.c.kirby@ttu.edu),                 |\n"
  << "|                      Mauro Perego (mperego@sandia.gov),                     |\n"
  << "|                      Kyungjoo Kim (kyukim@sandia.gov)                       |\n"
  << "|                                                                             |\n"
  << "|  Intrepid's website: http://trilinos.sandia.gov/packages/intrepid           |\n"
  << "|  Trilinos website:   http://trilinos.sandia.gov                             |\n"
  << "|                                                                             |\n"
  << "===============================================================================\n";

  typedef Kokkos::DynRankView<ValueType, DeviceType> DynRankView;
#define ConstructWithLabel(obj, ...) obj(#obj, __VA_ARGS__)

  const ValueType tol = tolerence();
  int errorFlag = 0;

  // for virtual function, value and point types are declared in the class
  typedef ValueType outputValueType;
  typedef ValueType pointValueType;
  typedef ValueType weightValueType;
  typedef Basis_HGRAD_TET_Cn_FEM_ORTH<DeviceType, outputValueType,
      pointValueType> tetBasisType;
  typedef CubatureDirectTetDefault<DeviceType, pointValueType,
      weightValueType> cubatureTetType;
  *outStream << "\n"
      << "===============================================================================\n"
      << "| TEST 1: constructors and exceptions                                         |\n"
      << "===============================================================================\n";


  constexpr ordinal_type maxOrder = Parameters::MaxOrder;
  constexpr ordinal_type dim = 3;
  try {
    const ordinal_type order = std::min(3, maxOrder);
    tetBasisType tetBasis(order);

    const ordinal_type polydim = tetBasis.getCardinality();

    // First, get a reference quadrature rule
    cubatureTetType cub(2 * order);
    const ordinal_type npts = cub.getNumPoints();
    DynRankView ConstructWithLabel(cubPoints, npts, dim);
    DynRankView ConstructWithLabel(cubWeights, npts);

    cub.getCubature(cubPoints, cubWeights);

    // Tabulate the basis functions at the cubature points
    DynRankView ConstructWithLabel(basisAtCubPts, polydim, npts);

    tetBasis.getValues(basisAtCubPts, cubPoints, OPERATOR_VALUE);

    auto h_basisAtCubPts = Kokkos::create_mirror_view(basisAtCubPts);
    Kokkos::deep_copy(h_basisAtCubPts, basisAtCubPts);

    auto h_cubWeights = Kokkos::create_mirror_view(cubWeights);
    Kokkos::deep_copy(h_cubWeights, cubWeights);

    // Now let's compute the mass matrix
    for (ordinal_type i = 0; i < polydim; i++) {
      for (ordinal_type j = i; j < polydim; j++) {
        ValueType cur = 0.0;
        for (ordinal_type k = 0; k < cub.getNumPoints(); k++) {
          cur += h_cubWeights(k) * h_basisAtCubPts(i, k)
							                * h_basisAtCubPts(j, k);
        }
        if (i != j && std::abs(cur) > tol) {
          errorFlag++;
        }
        if (i == j && std::abs(cur) < tol) {
          errorFlag++;
        }
      }
    }
  }
  catch (std::exception &err) {
    *outStream << err.what() << "\n\n";
    errorFlag = -1000;
  }


  try {
    constexpr ordinal_type order = 3;
    if(order <= maxOrder) {
    tetBasisType tetBasis(order);
    shards::CellTopology tet_4(
        shards::getCellTopologyData<shards::Tetrahedron<4> >());
    const ordinal_type np_lattice = PointTools::getLatticeSize(tet_4, order,
        0);
    const ordinal_type polydim = tetBasis.getCardinality();
    DynRankView ConstructWithLabel(lattice, np_lattice , dim);
    PointTools::getLattice(lattice, tet_4, order, 0, POINTTYPE_EQUISPACED);

    DynRankView ConstructWithLabel(dBasisAtLattice, polydim , np_lattice , dim);
    tetBasis.getValues(dBasisAtLattice, lattice, OPERATOR_D1);

    auto h_dBasisAtLattice = Kokkos::create_mirror_view(dBasisAtLattice);
    Kokkos::deep_copy(h_dBasisAtLattice, dBasisAtLattice);


    const double fiat_vals[] = {
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    5.477225575051661e+00, 2.738612787525831e+00, 2.738612787525831e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252568e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252568e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252568e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252568e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252568e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252568e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 4.743416490252569e+00, 1.581138830084190e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999579e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999579e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999579e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999579e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999579e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999579e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 4.472135954999580e+00,
    -3.074085229787880e+01, -1.024695076595960e+01, -1.024695076595960e+01,
    -1.024695076595960e+01, 0.000000000000000e+00, 0.000000000000000e+00,
    1.024695076595960e+01, 1.024695076595960e+01, 1.024695076595960e+01,
    3.074085229787880e+01, 2.049390153191920e+01, 2.049390153191920e+01,
    -2.049390153191920e+01, -6.831300510639732e+00, -6.831300510639732e+00,
    -1.706460100885310e-15, 3.415650255319865e+00, 3.415650255319865e+00,
    2.049390153191919e+01, 1.366260102127946e+01, 1.366260102127946e+01,
    -1.024695076595960e+01, -3.415650255319866e+00, -3.415650255319866e+00,
    1.024695076595960e+01, 6.831300510639730e+00, 6.831300510639730e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    -2.049390153191920e+01, -6.831300510639732e+00, -6.831300510639732e+00,
    -1.706460100885310e-15, 3.415650255319865e+00, 3.415650255319865e+00,
    2.049390153191919e+01, 1.366260102127946e+01, 1.366260102127946e+01,
    -1.024695076595960e+01, -3.415650255319866e+00, -3.415650255319866e+00,
    1.024695076595960e+01, 6.831300510639730e+00, 6.831300510639730e+00,
    -1.706460100885310e-15, -5.688200336284365e-16, -5.688200336284365e-16,
    -1.024695076595960e+01, -3.415650255319866e+00, -3.415650255319866e+00,
    1.024695076595960e+01, 6.831300510639730e+00, 6.831300510639730e+00,
    -1.706460100885310e-15, -5.688200336284365e-16, -5.688200336284365e-16,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    -7.937253933193772e+00, -2.381176179958132e+01, -7.937253933193772e+00,
    -7.937253933193772e+00, -1.058300524425836e+01, -5.291502622129182e+00,
    -7.937253933193772e+00, 2.645751311064589e+00, -2.645751311064591e+00,
    -7.937253933193772e+00, 1.587450786638754e+01, 0.000000000000000e+00,
    5.291502622129181e+00, -1.058300524425837e+01, -4.406061034464155e-16,
    5.291502622129181e+00, 2.645751311064589e+00, 2.645751311064590e+00,
    5.291502622129181e+00, 1.587450786638754e+01, 5.291502622129181e+00,
    1.852025917745213e+01, 2.645751311064588e+00, 7.937253933193770e+00,
    1.852025917745213e+01, 1.587450786638754e+01, 1.058300524425836e+01,
    3.174901573277509e+01, 1.587450786638754e+01, 1.587450786638754e+01,
    -5.291502622129182e+00, -1.587450786638755e+01, -5.291502622129182e+00,
    -5.291502622129182e+00, -2.645751311064592e+00, -2.645751311064591e+00,
    -5.291502622129182e+00, 1.058300524425836e+01, -8.812122068928310e-16,
    7.937253933193772e+00, -2.645751311064592e+00, 2.645751311064591e+00,
    7.937253933193772e+00, 1.058300524425836e+01, 5.291502622129181e+00,
    2.116601048851673e+01, 1.058300524425836e+01, 1.058300524425836e+01,
    -2.645751311064591e+00, -7.937253933193774e+00, -2.645751311064591e+00,
    -2.645751311064591e+00, 5.291502622129181e+00, -4.406061034464155e-16,
    1.058300524425836e+01, 5.291502622129181e+00, 5.291502622129181e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    -6.480740698407860e+00, -3.240370349203930e+00, -2.268259244442751e+01,
    -6.480740698407860e+00, -3.240370349203930e+00, -9.721111047611791e+00,
    -6.480740698407860e+00, -3.240370349203930e+00, 3.240370349203929e+00,
    -6.480740698407860e+00, -3.240370349203930e+00, 1.620185174601965e+01,
    -6.480740698407860e+00, -3.240370349203930e+00, -1.620185174601965e+01,
    -6.480740698407860e+00, -3.240370349203930e+00, -3.240370349203932e+00,
    -6.480740698407860e+00, -3.240370349203930e+00, 9.721111047611787e+00,
    -6.480740698407860e+00, -3.240370349203930e+00, -9.721111047611791e+00,
    -6.480740698407860e+00, -3.240370349203930e+00, 3.240370349203929e+00,
    -6.480740698407860e+00, -3.240370349203930e+00, -3.240370349203930e+00,
    6.480740698407860e+00, 3.240370349203930e+00, -9.721111047611791e+00,
    6.480740698407860e+00, 3.240370349203930e+00, 3.240370349203929e+00,
    6.480740698407860e+00, 3.240370349203930e+00, 1.620185174601965e+01,
    6.480740698407860e+00, 3.240370349203930e+00, -3.240370349203930e+00,
    6.480740698407860e+00, 3.240370349203930e+00, 9.721111047611789e+00,
    6.480740698407860e+00, 3.240370349203930e+00, 3.240370349203929e+00,
    1.944222209522358e+01, 9.721111047611791e+00, 3.240370349203930e+00,
    1.944222209522358e+01, 9.721111047611791e+00, 1.620185174601965e+01,
    1.944222209522358e+01, 9.721111047611791e+00, 9.721111047611789e+00,
    3.240370349203930e+01, 1.620185174601965e+01, 1.620185174601965e+01,
    0.000000000000000e+00, -1.833030277982336e+01, -4.582575694955840e+00,
    0.000000000000000e+00, -1.833030277982336e+01, -4.582575694955840e+00,
    0.000000000000000e+00, -1.833030277982336e+01, -4.582575694955839e+00,
    0.000000000000000e+00, -1.833030277982336e+01, -4.582575694955840e+00,
    0.000000000000000e+00, -3.055050463303894e+00, 1.527525231651946e+00,
    0.000000000000000e+00, -3.055050463303894e+00, 1.527525231651946e+00,
    0.000000000000000e+00, -3.055050463303894e+00, 1.527525231651946e+00,
    0.000000000000000e+00, 1.222020185321557e+01, 7.637626158259732e+00,
    0.000000000000000e+00, 1.222020185321557e+01, 7.637626158259732e+00,
    0.000000000000000e+00, 2.749545416973504e+01, 1.374772708486752e+01,
    0.000000000000000e+00, -1.222020185321557e+01, -3.055050463303894e+00,
    0.000000000000000e+00, -1.222020185321557e+01, -3.055050463303894e+00,
    0.000000000000000e+00, -1.222020185321557e+01, -3.055050463303894e+00,
    0.000000000000000e+00, 3.055050463303893e+00, 3.055050463303893e+00,
    0.000000000000000e+00, 3.055050463303893e+00, 3.055050463303893e+00,
    0.000000000000000e+00, 1.833030277982336e+01, 9.165151389911680e+00,
    0.000000000000000e+00, -6.110100926607787e+00, -1.527525231651947e+00,
    0.000000000000000e+00, -6.110100926607787e+00, -1.527525231651947e+00,
    0.000000000000000e+00, 9.165151389911678e+00, 4.582575694955839e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, -5.612486080160912e+00, -1.309580085370879e+01,
    0.000000000000000e+00, -5.612486080160912e+00, -1.309580085370879e+01,
    0.000000000000000e+00, -5.612486080160911e+00, -1.309580085370879e+01,
    0.000000000000000e+00, -5.612486080160912e+00, -1.309580085370879e+01,
    0.000000000000000e+00, -5.612486080160912e+00, -1.870828693386971e+00,
    0.000000000000000e+00, -5.612486080160912e+00, -1.870828693386971e+00,
    0.000000000000000e+00, -5.612486080160912e+00, -1.870828693386971e+00,
    0.000000000000000e+00, -5.612486080160911e+00, 9.354143466934852e+00,
    0.000000000000000e+00, -5.612486080160911e+00, 9.354143466934852e+00,
    0.000000000000000e+00, -5.612486080160912e+00, 2.057911562725668e+01,
    0.000000000000000e+00, 5.612486080160912e+00, -5.612486080160912e+00,
    0.000000000000000e+00, 5.612486080160912e+00, -5.612486080160912e+00,
    0.000000000000000e+00, 5.612486080160912e+00, -5.612486080160912e+00,
    0.000000000000000e+00, 5.612486080160912e+00, 5.612486080160912e+00,
    0.000000000000000e+00, 5.612486080160912e+00, 5.612486080160912e+00,
    0.000000000000000e+00, 5.612486080160912e+00, 1.683745824048274e+01,
    0.000000000000000e+00, 1.683745824048273e+01, 1.870828693386970e+00,
    0.000000000000000e+00, 1.683745824048273e+01, 1.870828693386970e+00,
    0.000000000000000e+00, 1.683745824048273e+01, 1.309580085370879e+01,
    0.000000000000000e+00, 2.806243040080456e+01, 9.354143466934854e+00,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -8.812122068928310e-16,
    0.000000000000000e+00, 0.000000000000000e+00, -8.812122068928310e-16,
    0.000000000000000e+00, 0.000000000000000e+00, -8.812122068928310e-16,
    0.000000000000000e+00, 0.000000000000000e+00, -8.812122068928310e-16,
    0.000000000000000e+00, 0.000000000000000e+00, -8.812122068928310e-16,
    0.000000000000000e+00, 0.000000000000000e+00, -8.812122068928310e-16,
    0.000000000000000e+00, 0.000000000000000e+00, 1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 1.322875655532295e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.645751311064591e+01,
    9.524704719832526e+01, 2.381176179958132e+01, 2.381176179958132e+01,
    -1.058300524425836e+01, -1.322875655532295e+01, -1.322875655532295e+01,
    -1.058300524425837e+01, 2.645751311064586e+00, 2.645751311064586e+00,
    9.524704719832526e+01, 7.143528539874394e+01, 7.143528539874394e+01,
    4.233202097703347e+01, 1.058300524425837e+01, 1.058300524425837e+01,
    -1.058300524425836e+01, -5.291502622129184e+00, -5.291502622129184e+00,
    4.233202097703344e+01, 3.174901573277508e+01, 3.174901573277508e+01,
    1.058300524425837e+01, 2.645751311064592e+00, 2.645751311064592e+00,
    1.058300524425836e+01, 7.937253933193769e+00, 7.937253933193769e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    4.233202097703347e+01, 1.058300524425837e+01, 1.058300524425837e+01,
    -1.058300524425836e+01, -5.291502622129184e+00, -5.291502622129184e+00,
    4.233202097703344e+01, 3.174901573277508e+01, 3.174901573277508e+01,
    1.058300524425837e+01, 2.645751311064592e+00, 2.645751311064592e+00,
    1.058300524425836e+01, 7.937253933193769e+00, 7.937253933193769e+00,
    2.935026245019504e-31, 7.337565612548760e-32, 7.337565612548760e-32,
    1.058300524425837e+01, 2.645751311064592e+00, 2.645751311064592e+00,
    1.058300524425836e+01, 7.937253933193769e+00, 7.937253933193769e+00,
    2.935026245019504e-31, 7.337565612548760e-32, 7.337565612548760e-32,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    4.024922359499622e+01, 6.037383539249433e+01, 2.012461179749811e+01,
    1.341640786499874e+01, -1.565247584249853e+01, -2.236067977499790e+00,
    -1.341640786499874e+01, -2.906888370749726e+01, -1.565247584249853e+01,
    -4.024922359499622e+01, 2.012461179749811e+01, -2.012461179749811e+01,
    -3.577708763999664e+01, 8.944271909999161e+00, -8.944271909999159e+00,
    -2.979040983896728e-15, -4.472135954999583e+00, 4.472135954999577e+00,
    3.577708763999663e+01, 4.472135954999577e+01, 2.683281572999747e+01,
    -4.919349550499538e+01, -1.118033988749895e+01, -1.565247584249853e+01,
    4.919349550499536e+01, 3.801315561749642e+01, 3.354101966249684e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    1.788854381999832e+01, 2.683281572999748e+01, 8.944271909999161e+00,
    1.489520491948364e-15, -1.341640786499874e+01, -4.472135954999580e+00,
    -1.788854381999832e+01, 8.944271909999145e+00, -8.944271909999161e+00,
    -2.236067977499789e+01, -2.236067977499788e+00, -6.708203932499368e+00,
    2.236067977499789e+01, 2.012461179749810e+01, 1.565247584249852e+01,
    -8.937122951690183e-15, -2.979040983896728e-15, -2.979040983896728e-15,
    4.472135954999580e+00, 6.708203932499371e+00, 2.236067977499790e+00,
    -4.472135954999580e+00, 2.236067977499787e+00, -2.236067977499790e+00,
    -4.468561475845092e-15, -1.489520491948364e-15, -1.489520491948364e-15,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    3.485685011586675e+01, 1.161895003862225e+01, 5.809475019311125e+01,
    1.161895003862225e+01, 0.000000000000000e+00, -1.549193338482967e+01,
    -1.161895003862225e+01, -1.161895003862225e+01, -2.711088342345191e+01,
    -3.485685011586675e+01, -2.323790007724450e+01, 2.323790007724450e+01,
    2.323790007724450e+01, 7.745966692414833e+00, 2.840187787218773e+01,
    1.934943878227166e-15, -3.872983346207416e+00, -1.420093893609386e+01,
    -2.323790007724450e+01, -1.549193338482966e+01, 5.163977794943212e+00,
    1.161895003862225e+01, 3.872983346207417e+00, 9.036961141150641e+00,
    -1.161895003862225e+01, -7.745966692414831e+00, -2.581988897471612e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    -3.872983346207416e+01, -1.290994448735805e+01, 7.745966692414837e+00,
    -3.224906463711944e-15, 6.454972243679026e+00, -3.872983346207421e+00,
    3.872983346207415e+01, 2.581988897471610e+01, 4.647580015448898e+01,
    -1.936491673103708e+01, -6.454972243679027e+00, -1.290994448735804e+00,
    1.936491673103708e+01, 1.290994448735805e+01, 1.807392228230127e+01,
    -3.224906463711944e-15, -1.074968821237314e-15, -1.074968821237314e-15,
    -5.034878350069641e+01, -1.678292783356547e+01, -1.161895003862225e+01,
    5.034878350069641e+01, 3.356585566713093e+01, 3.872983346207415e+01,
    -8.384756805651052e-15, -2.794918935217017e-15, -2.794918935217017e-15,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    1.039230484541326e+01, 6.754998149518620e+01, 1.558845726811989e+01,
    1.039230484541326e+01, 2.598076211353317e+01, 8.660254037844389e+00,
    1.039230484541326e+01, -1.558845726811989e+01, 1.732050807568879e+00,
    1.039230484541326e+01, -5.715767664977296e+01, -5.196152422706632e+00,
    -6.928203230275511e+00, -1.039230484541326e+01, -1.039230484541326e+01,
    -6.928203230275511e+00, -3.464101615137756e+00, -3.464101615137756e+00,
    -6.928203230275511e+00, 3.464101615137752e+00, 3.464101615137753e+00,
    2.424871130596427e+01, -1.558845726811990e+01, 1.732050807568872e+00,
    2.424871130596427e+01, 3.983716857408416e+01, 2.251666049839540e+01,
    1.039230484541326e+02, 5.196152422706632e+01, 5.196152422706632e+01,
    4.618802153517007e+00, 3.002221399786055e+01, 6.928203230275511e+00,
    4.618802153517007e+00, 2.309401076758505e+00, 2.309401076758504e+00,
    4.618802153517007e+00, -2.540341184434353e+01, -2.309401076758502e+00,
    1.154700538379250e+00, -9.814954576223640e+00, -4.041451884327382e+00,
    1.154700538379250e+00, 1.096965511460288e+01, 5.196152422706630e+00,
    4.618802153517006e+01, 2.309401076758502e+01, 2.309401076758503e+01,
    1.154700538379252e+00, 7.505553499465138e+00, 1.732050807568878e+00,
    1.154700538379252e+00, -6.350852961085883e+00, -5.773502691896256e-01,
    1.154700538379251e+01, 5.773502691896255e+00, 5.773502691896256e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    9.000000000000000e+00, 2.700000000000000e+01, 4.500000000000000e+01,
    9.000000000000000e+00, 1.200000000000000e+01, 1.800000000000000e+01,
    9.000000000000000e+00, -2.999999999999998e+00, -8.999999999999996e+00,
    9.000000000000000e+00, -1.800000000000000e+01, -3.600000000000000e+01,
    -6.000000000000000e+00, 1.200000000000000e+01, -1.600000000000000e+01,
    -6.000000000000000e+00, -2.999999999999998e+00, -3.000000000000001e+00,
    -6.000000000000000e+00, -1.800000000000000e+01, 9.999999999999996e+00,
    -2.100000000000000e+01, -2.999999999999997e+00, -3.700000000000000e+01,
    -2.100000000000000e+01, -1.800000000000000e+01, 1.599999999999999e+01,
    -3.600000000000000e+01, -1.800000000000000e+01, -1.800000000000000e+01,
    -1.000000000000000e+01, -3.000000000000001e+01, 6.000000000000003e+00,
    -1.000000000000000e+01, -5.000000000000002e+00, -4.999999999999999e+00,
    -1.000000000000000e+01, 1.999999999999999e+01, -1.600000000000000e+01,
    1.500000000000000e+01, -5.000000000000002e+00, -7.000000000000002e+00,
    1.500000000000000e+01, 1.999999999999999e+01, 2.199999999999999e+01,
    3.999999999999999e+01, 1.999999999999999e+01, 1.999999999999999e+01,
    -1.300000000000000e+01, -3.900000000000001e+01, -8.999999999999998e+00,
    -1.300000000000000e+01, 2.599999999999999e+01, -4.000000000000002e+00,
    5.199999999999999e+01, 2.599999999999999e+01, 2.599999999999999e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    7.348469228349536e+00, 3.674234614174768e+00, 5.511351921262152e+01,
    7.348469228349536e+00, 3.674234614174768e+00, 2.082066281365702e+01,
    7.348469228349536e+00, 3.674234614174768e+00, -1.347219358530748e+01,
    7.348469228349536e+00, 3.674234614174768e+00, -4.776504998427198e+01,
    7.348469228349536e+00, 3.674234614174768e+00, 3.796709101313927e+01,
    7.348469228349536e+00, 3.674234614174768e+00, 3.674234614174771e+00,
    7.348469228349536e+00, 3.674234614174768e+00, -3.061862178478972e+01,
    7.348469228349536e+00, 3.674234614174768e+00, 2.082066281365702e+01,
    7.348469228349536e+00, 3.674234614174768e+00, -1.347219358530748e+01,
    7.348469228349536e+00, 3.674234614174768e+00, 3.674234614174768e+00,
    -4.082482904638631e+00, -2.041241452319316e+00, -1.347219358530748e+01,
    -4.082482904638631e+00, -2.041241452319316e+00, -2.041241452319317e+00,
    -4.082482904638631e+00, -2.041241452319316e+00, 9.389710680668845e+00,
    -4.082482904638631e+00, -2.041241452319316e+00, -7.756717518813399e+00,
    -4.082482904638631e+00, -2.041241452319316e+00, 3.674234614174765e+00,
    -4.082482904638631e+00, -2.041241452319316e+00, -2.041241452319317e+00,
    3.021037349432586e+01, 1.510518674716293e+01, -1.347219358530748e+01,
    3.021037349432586e+01, 1.510518674716293e+01, 4.368256707963334e+01,
    3.021037349432586e+01, 1.510518674716293e+01, 1.510518674716292e+01,
    1.102270384252430e+02, 5.511351921262151e+01, 5.511351921262151e+01,
    0.000000000000000e+00, 4.500000000000001e+01, 9.000000000000000e+00,
    0.000000000000000e+00, 4.500000000000001e+01, 9.000000000000000e+00,
    0.000000000000000e+00, 4.499999999999999e+01, 9.000000000000000e+00,
    0.000000000000000e+00, 4.500000000000001e+01, 9.000000000000000e+00,
    0.000000000000000e+00, -1.000000000000000e+01, -5.999999999999999e+00,
    0.000000000000000e+00, -1.000000000000000e+01, -5.999999999999999e+00,
    0.000000000000000e+00, -1.000000000000000e+01, -5.999999999999999e+00,
    0.000000000000000e+00, 4.999999999999996e+00, 8.999999999999996e+00,
    0.000000000000000e+00, 4.999999999999996e+00, 8.999999999999996e+00,
    0.000000000000000e+00, 9.000000000000000e+01, 5.400000000000000e+01,
    0.000000000000000e+00, 2.000000000000000e+01, 4.000000000000000e+00,
    0.000000000000000e+00, 2.000000000000000e+01, 4.000000000000000e+00,
    0.000000000000000e+00, 2.000000000000000e+01, 4.000000000000000e+00,
    0.000000000000000e+00, -5.000000000000002e+00, -1.000000000000001e+00,
    0.000000000000000e+00, -5.000000000000002e+00, -1.000000000000001e+00,
    0.000000000000000e+00, 4.000000000000000e+01, 2.400000000000000e+01,
    0.000000000000000e+00, 5.000000000000000e+00, 1.000000000000000e+00,
    0.000000000000000e+00, 5.000000000000000e+00, 1.000000000000000e+00,
    0.000000000000000e+00, 9.999999999999996e+00, 5.999999999999998e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 2.078460969082653e+01, 2.598076211353316e+01,
    0.000000000000000e+00, 2.078460969082653e+01, 2.598076211353316e+01,
    0.000000000000000e+00, 2.078460969082652e+01, 2.598076211353316e+01,
    0.000000000000000e+00, 2.078460969082653e+01, 2.598076211353316e+01,
    0.000000000000000e+00, 3.464101615137756e+00, -1.327905619136139e+01,
    0.000000000000000e+00, 3.464101615137756e+00, -1.327905619136139e+01,
    0.000000000000000e+00, 3.464101615137756e+00, -1.327905619136139e+01,
    0.000000000000000e+00, -1.385640646055102e+01, -6.350852961085886e+00,
    0.000000000000000e+00, -1.385640646055102e+01, -6.350852961085886e+00,
    0.000000000000000e+00, -3.117691453623979e+01, 4.676537180435969e+01,
    0.000000000000000e+00, -2.309401076758503e+01, 3.464101615137756e+00,
    0.000000000000000e+00, -2.309401076758503e+01, 3.464101615137756e+00,
    0.000000000000000e+00, -2.309401076758503e+01, 3.464101615137756e+00,
    0.000000000000000e+00, 5.773502691896256e+00, 1.154700538379249e+00,
    0.000000000000000e+00, 5.773502691896256e+00, 1.154700538379249e+00,
    0.000000000000000e+00, 3.464101615137754e+01, 4.503332099679081e+01,
    0.000000000000000e+00, -3.002221399786054e+01, -5.196152422706631e+00,
    0.000000000000000e+00, -3.002221399786054e+01, -5.196152422706631e+00,
    0.000000000000000e+00, 4.503332099679080e+01, 2.944486372867091e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 0.000000000000000e+00,
    0.000000000000000e+00, 6.363961030678929e+00, 3.181980515339464e+01,
    0.000000000000000e+00, 6.363961030678929e+00, 3.181980515339464e+01,
    0.000000000000000e+00, 6.363961030678928e+00, 3.181980515339464e+01,
    0.000000000000000e+00, 6.363961030678929e+00, 3.181980515339464e+01,
    0.000000000000000e+00, 6.363961030678929e+00, 2.121320343559643e+00,
    0.000000000000000e+00, 6.363961030678929e+00, 2.121320343559643e+00,
    0.000000000000000e+00, 6.363961030678929e+00, 2.121320343559643e+00,
    0.000000000000000e+00, 6.363961030678928e+00, -2.757716446627535e+01,
    0.000000000000000e+00, 6.363961030678928e+00, -2.757716446627535e+01,
    0.000000000000000e+00, 6.363961030678929e+00, -5.727564927611036e+01,
    0.000000000000000e+00, -3.535533905932738e+00, -7.778174593052023e+00,
    0.000000000000000e+00, -3.535533905932738e+00, -7.778174593052023e+00,
    0.000000000000000e+00, -3.535533905932738e+00, -7.778174593052023e+00,
    0.000000000000000e+00, -3.535533905932738e+00, 2.121320343559642e+00,
    0.000000000000000e+00, -3.535533905932738e+00, 2.121320343559642e+00,
    0.000000000000000e+00, -3.535533905932738e+00, 1.202081528017131e+01,
    0.000000000000000e+00, 2.616295090390225e+01, -7.778174593052024e+00,
    0.000000000000000e+00, 2.616295090390225e+01, -7.778174593052024e+00,
    0.000000000000000e+00, 2.616295090390225e+01, 4.171930009000628e+01,
    0.000000000000000e+00, 9.545941546018391e+01, 3.181980515339464e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000001e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000001e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000000e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000001e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000001e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000001e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000001e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000000e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000000e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 2.700000000000001e+01,
    0.000000000000000e+00, 0.000000000000000e+00, -8.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, -8.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, -8.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, -8.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, -8.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, -8.000000000000000e+00,
    0.000000000000000e+00, 0.000000000000000e+00, 1.299999999999999e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 1.299999999999999e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 1.299999999999999e+01,
    0.000000000000000e+00, 0.000000000000000e+00, 9.000000000000000e+01
    };

    int fiat_index_cur = 0;
    for (ordinal_type i=0;i<polydim;i++) {
      for (ordinal_type j=0;j<np_lattice;j++) {
        for (ordinal_type k=0;k<dim;k++) {
          if (std::abs( h_dBasisAtLattice(i,j,k) - fiat_vals[fiat_index_cur] ) > 10.0*tol ) {
            errorFlag++;
            *outStream << std::setw(70) << "^^^^----FAILURE!" << "\n";

            // Output the multi-index of the value where the error is:
            *outStream << " At multi-index { ";
            *outStream << i << " " << j << " " << k;
            *outStream << "}  computed value: " << h_dBasisAtLattice(i,j,k)
                                     << " but correct value: " << fiat_vals[fiat_index_cur] << "\n";
            *outStream << "Difference: " << std::abs( h_dBasisAtLattice(i,j,k) - fiat_vals[fiat_index_cur] ) << "\n";
          }
          fiat_index_cur++;
        }
      }
    }
    }
  }  catch (std::exception &err) {
    *outStream << err.what() << "\n\n";
    errorFlag = -1000;
  }

  *outStream
  << "\n"
  << "===============================================================================\n"
  << "| TEST 2: Function Space is Correct                                           |\n"
  << "===============================================================================\n";
  
  try {
    const ordinal_type order = std::min(3, maxOrder);
    tetBasisType tetBasis(order);
    
    const EFunctionSpace fs = tetBasis.getFunctionSpace();
    
    if (fs != FUNCTION_SPACE_HGRAD)
    {
      *outStream << std::setw(70) << "------------- TEST FAILURE! -------------" << "\n";
      
      // Output the multi-index of the value where the error is:
      *outStream << " Expected a function space of FUNCTION_SPACE_HGRAD (enum value " << FUNCTION_SPACE_HGRAD << "),";
      *outStream << " but got " << fs << "\n";
      if (fs == FUNCTION_SPACE_MAX)
      {
        *outStream << "Note that this matches the default value defined by superclass, FUNCTION_SPACE_MAX.  Likely the subclass has failed to set the superclass functionSpace_ field.\n";
      }
      errorFlag++;
    }
  } catch (std::logic_error &err){
    *outStream << "UNEXPECTED ERROR !!! ----------------------------------------------------------\n";
    *outStream << err.what() << '\n';
    *outStream << "-------------------------------------------------------------------------------" << "\n\n";
    errorFlag = -1000;
  }

  // second order derivatives test missing!!
  
  if (errorFlag != 0)
    std::cout << "End Result: TEST FAILED\n";
  else
    std::cout << "End Result: TEST PASSED\n";

  // reset format state of std::cout
  std::cout.copyfmt(oldFormatState);
  return errorFlag;

}

}
}




