//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// G4MaterialScanner implementation
//
// Author: M.Asai, May 2006
// --------------------------------------------------------------------

#include "G4MaterialScanner.hh"

#include "G4Event.hh"
#include "G4EventManager.hh"
#include "G4GeometryManager.hh"
#include "G4MSSteppingAction.hh"
#include "G4MatScanMessenger.hh"
#include "G4ProcessManager.hh"
#include "G4ProcessVector.hh"
#include "G4RayShooter.hh"
#include "G4Region.hh"
#include "G4RegionStore.hh"
#include "G4RunManagerKernel.hh"
#include "G4SDManager.hh"
#include "G4StateManager.hh"
#include "G4SystemOfUnits.hh"
#include "G4TransportationManager.hh"

// --------------------------------------------------------------------
G4MaterialScanner::G4MaterialScanner()
{
  theRayShooter = new G4RayShooter();
  theMessenger = new G4MatScanMessenger(this);
  theEventManager = G4EventManager::GetEventManager();

  eyePosition = G4ThreeVector(0., 0., 0.);
  thetaSpan = 90. * deg;
  phiSpan = 360. * deg;
}

// --------------------------------------------------------------------
G4MaterialScanner::~G4MaterialScanner()
{
  delete theRayShooter;
  delete theMatScannerSteppingAction;
  delete theMessenger;
}

// --------------------------------------------------------------------
void G4MaterialScanner::Scan()
{
  G4StateManager* theStateMan = G4StateManager::GetStateManager();
  G4ApplicationState currentState = theStateMan->GetCurrentState();
  if (currentState != G4State_Idle) {
    G4cerr << "Illegal application state - Scan() ignored." << G4endl;
    return;
  }

  if (theMatScannerSteppingAction == nullptr) {
    theMatScannerSteppingAction = new G4MSSteppingAction();
  }
  StoreUserActions();
  DoScan();
  RestoreUserActions();
}

// --------------------------------------------------------------------
void G4MaterialScanner::StoreUserActions()
{
  theUserEventAction = theEventManager->GetUserEventAction();
  theUserStackingAction = theEventManager->GetUserStackingAction();
  theUserTrackingAction = theEventManager->GetUserTrackingAction();
  theUserSteppingAction = theEventManager->GetUserSteppingAction();

  theEventManager->SetUserAction(theMatScannerEventAction);
  theEventManager->SetUserAction(theMatScannerStackingAction);
  theEventManager->SetUserAction(theMatScannerTrackingAction);
  theEventManager->SetUserAction(theMatScannerSteppingAction);

  G4SDManager* theSDMan = G4SDManager::GetSDMpointerIfExist();
  if (theSDMan != nullptr) {
    theSDMan->Activate("/", false);
  }

  G4GeometryManager* theGeomMan = G4GeometryManager::GetInstance();
  theGeomMan->OpenGeometry();
  theGeomMan->CloseGeometry(true);
}

// --------------------------------------------------------------------
void G4MaterialScanner::RestoreUserActions()
{
  theEventManager->SetUserAction(theUserEventAction);
  theEventManager->SetUserAction(theUserStackingAction);
  theEventManager->SetUserAction(theUserTrackingAction);
  theEventManager->SetUserAction(theUserSteppingAction);

  G4SDManager* theSDMan = G4SDManager::GetSDMpointerIfExist();
  if (theSDMan != nullptr) {
    theSDMan->Activate("/", true);
  }
}

// --------------------------------------------------------------------
void G4MaterialScanner::DoScan()
{
  // Confirm material table is updated
  G4RunManagerKernel::GetRunManagerKernel()->UpdateRegion();

  // Close geometry and set the application state
  G4GeometryManager* geomManager = G4GeometryManager::GetInstance();
  geomManager->OpenGeometry();
  geomManager->CloseGeometry(true, false);

  G4ThreeVector center(0, 0, 0);
  G4Navigator* navigator =
    G4TransportationManager::GetTransportationManager()->GetNavigatorForTracking();
  navigator->LocateGlobalPointAndSetup(center, nullptr, false);

  G4StateManager* theStateMan = G4StateManager::GetStateManager();
  theStateMan->SetNewState(G4State_GeomClosed);

  // Event loop
  G4int iEvent = 0;
  for (G4int iTheta = 0; iTheta < nTheta; ++iTheta) {
    G4double theta = thetaMin;
    if (iTheta > 0) theta += G4double(iTheta) * thetaSpan / G4double(nTheta - 1);
    G4double aveLength = 0.;
    G4double aveX0 = 0.;
    G4double aveLambda = 0.;
    G4cout << G4endl;
    G4cout << "         Theta(deg)    Phi(deg)  Length(mm)          x0     lambda0" << G4endl;
    G4cout << G4endl;
    for (G4int iPhi = 0; iPhi < nPhi; ++iPhi) {
      auto anEvent = new G4Event(iEvent++);
      G4double phi = phiMin;
      if (iPhi > 0) phi += G4double(iPhi) * phiSpan / G4double(nPhi - 1);
      eyeDirection = G4ThreeVector(std::cos(theta) * std::cos(phi), std::cos(theta) * std::sin(phi),
                                   std::sin(theta));
      theRayShooter->Shoot(anEvent, eyePosition, eyeDirection);
      theMatScannerSteppingAction->Initialize(regionSensitive, theRegion);
      theEventManager->ProcessOneEvent(anEvent);
      G4double length = theMatScannerSteppingAction->GetTotalStepLength();
      G4double x0 = theMatScannerSteppingAction->GetX0();
      G4double lambda = theMatScannerSteppingAction->GetLambda0();

      G4cout << "        " << std::setw(11) << theta / deg << " " << std::setw(11) << phi / deg
             << " " << std::setw(11) << length / mm << " " << std::setw(11) << x0 << " "
             << std::setw(11) << lambda;
      if(1 == verbosity)
      {
        theMatScannerSteppingAction->PrintIntegratedMaterialVerbose(G4cout);
      }
      else if(2 == verbosity)
      {
        theMatScannerSteppingAction->PrintEachMaterialVerbose(G4cout);
      }
      G4cout << G4endl;
      aveLength += length / mm;
      aveX0 += x0;
      aveLambda += lambda;
    }
    if (nPhi > 1) {
      G4cout << G4endl;
      G4cout << " ave. for theta = " << std::setw(11) << theta / deg << " : " << std::setw(11)
             << aveLength / nPhi << " " << std::setw(11) << aveX0 / nPhi << " " << std::setw(11)
             << aveLambda / nPhi << G4endl;
    }
  }

  theStateMan->SetNewState(G4State_Idle);
  return;
}

// --------------------------------------------------------------------
G4bool G4MaterialScanner::SetRegionName(const G4String& val)
{
  G4Region* aRegion = G4RegionStore::GetInstance()->GetRegion(val);
  if (aRegion != nullptr) {
    theRegion = aRegion;
    regionName = val;
    return true;
  }

  G4cerr << "Region <" << val << "> not found. Command ignored." << G4endl;
  G4cerr << "Defined regions are : " << G4endl;
  for (const auto& i : *G4RegionStore::GetInstance()) {
    G4cerr << " " << i->GetName();
  }
  G4cerr << G4endl;
  return false;
}
