// Copyright 2012-2015 Oliver Eilhard. All rights reserved.
// Use of this source code is governed by a MIT-license.
// See http://olivere.mit-license.org/license.txt for details.

package elastic

// MinAggregation is a single-value metrics aggregation that keeps track and
// returns the minimum value among numeric values extracted from the
// aggregated documents. These values can be extracted either from
// specific numeric fields in the documents, or be generated by a
// provided script.
// See: http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/search-aggregations-metrics-min-aggregation.html
type MinAggregation struct {
	field           string
	script          string
	scriptFile      string
	lang            string
	format          string
	params          map[string]interface{}
	subAggregations map[string]Aggregation
}

func NewMinAggregation() MinAggregation {
	a := MinAggregation{
		params:          make(map[string]interface{}),
		subAggregations: make(map[string]Aggregation),
	}
	return a
}

func (a MinAggregation) Field(field string) MinAggregation {
	a.field = field
	return a
}

func (a MinAggregation) Script(script string) MinAggregation {
	a.script = script
	return a
}

func (a MinAggregation) ScriptFile(scriptFile string) MinAggregation {
	a.scriptFile = scriptFile
	return a
}

func (a MinAggregation) Lang(lang string) MinAggregation {
	a.lang = lang
	return a
}

func (a MinAggregation) Format(format string) MinAggregation {
	a.format = format
	return a
}

func (a MinAggregation) Param(name string, value interface{}) MinAggregation {
	a.params[name] = value
	return a
}

func (a MinAggregation) SubAggregation(name string, subAggregation Aggregation) MinAggregation {
	a.subAggregations[name] = subAggregation
	return a
}

func (a MinAggregation) Source() interface{} {
	// Example:
	//	{
	//    "aggs" : {
	//      "min_price" : { "min" : { "field" : "price" } }
	//    }
	//	}
	// This method returns only the { "min" : { "field" : "price" } } part.

	source := make(map[string]interface{})
	opts := make(map[string]interface{})
	source["min"] = opts

	// ValuesSourceAggregationBuilder
	if a.field != "" {
		opts["field"] = a.field
	}
	if a.script != "" {
		opts["script"] = a.script
	}
	if a.scriptFile != "" {
		opts["script_file"] = a.scriptFile
	}
	if a.lang != "" {
		opts["lang"] = a.lang
	}
	if a.format != "" {
		opts["format"] = a.format
	}
	if len(a.params) > 0 {
		opts["params"] = a.params
	}

	// AggregationBuilder (SubAggregations)
	if len(a.subAggregations) > 0 {
		aggsMap := make(map[string]interface{})
		source["aggregations"] = aggsMap
		for name, aggregate := range a.subAggregations {
			aggsMap[name] = aggregate.Source()
		}
	}

	return source
}
