﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/accessanalyzer/AccessAnalyzerRequest.h>
#include <aws/accessanalyzer/AccessAnalyzer_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace AccessAnalyzer {
namespace Model {

/**
 * <p>Retrieves an analyzed resource.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/GetAnalyzedResourceRequest">AWS
 * API Reference</a></p>
 */
class GetAnalyzedResourceRequest : public AccessAnalyzerRequest {
 public:
  AWS_ACCESSANALYZER_API GetAnalyzedResourceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetAnalyzedResource"; }

  AWS_ACCESSANALYZER_API Aws::String SerializePayload() const override;

  AWS_ACCESSANALYZER_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources">ARN
   * of the analyzer</a> to retrieve information from.</p>
   */
  inline const Aws::String& GetAnalyzerArn() const { return m_analyzerArn; }
  inline bool AnalyzerArnHasBeenSet() const { return m_analyzerArnHasBeenSet; }
  template <typename AnalyzerArnT = Aws::String>
  void SetAnalyzerArn(AnalyzerArnT&& value) {
    m_analyzerArnHasBeenSet = true;
    m_analyzerArn = std::forward<AnalyzerArnT>(value);
  }
  template <typename AnalyzerArnT = Aws::String>
  GetAnalyzedResourceRequest& WithAnalyzerArn(AnalyzerArnT&& value) {
    SetAnalyzerArn(std::forward<AnalyzerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the resource to retrieve information about.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  GetAnalyzedResourceRequest& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_analyzerArn;
  bool m_analyzerArnHasBeenSet = false;

  Aws::String m_resourceArn;
  bool m_resourceArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace AccessAnalyzer
}  // namespace Aws
