﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/accessanalyzer/AccessAnalyzerRequest.h>
#include <aws/accessanalyzer/AccessAnalyzer_EXPORTS.h>
#include <aws/accessanalyzer/model/Criterion.h>
#include <aws/accessanalyzer/model/SortCriteria.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace AccessAnalyzer {
namespace Model {

/**
 * <p>Retrieves a list of findings generated by the specified
 * analyzer.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/ListFindingsRequest">AWS
 * API Reference</a></p>
 */
class ListFindingsRequest : public AccessAnalyzerRequest {
 public:
  AWS_ACCESSANALYZER_API ListFindingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListFindings"; }

  AWS_ACCESSANALYZER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources">ARN
   * of the analyzer</a> to retrieve findings from.</p>
   */
  inline const Aws::String& GetAnalyzerArn() const { return m_analyzerArn; }
  inline bool AnalyzerArnHasBeenSet() const { return m_analyzerArnHasBeenSet; }
  template <typename AnalyzerArnT = Aws::String>
  void SetAnalyzerArn(AnalyzerArnT&& value) {
    m_analyzerArnHasBeenSet = true;
    m_analyzerArn = std::forward<AnalyzerArnT>(value);
  }
  template <typename AnalyzerArnT = Aws::String>
  ListFindingsRequest& WithAnalyzerArn(AnalyzerArnT&& value) {
    SetAnalyzerArn(std::forward<AnalyzerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter to match for the findings to return.</p>
   */
  inline const Aws::Map<Aws::String, Criterion>& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = Aws::Map<Aws::String, Criterion>>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = Aws::Map<Aws::String, Criterion>>
  ListFindingsRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  template <typename FilterKeyT = Aws::String, typename FilterValueT = Criterion>
  ListFindingsRequest& AddFilter(FilterKeyT&& key, FilterValueT&& value) {
    m_filterHasBeenSet = true;
    m_filter.emplace(std::forward<FilterKeyT>(key), std::forward<FilterValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The sort order for the findings returned.</p>
   */
  inline const SortCriteria& GetSort() const { return m_sort; }
  inline bool SortHasBeenSet() const { return m_sortHasBeenSet; }
  template <typename SortT = SortCriteria>
  void SetSort(SortT&& value) {
    m_sortHasBeenSet = true;
    m_sort = std::forward<SortT>(value);
  }
  template <typename SortT = SortCriteria>
  ListFindingsRequest& WithSort(SortT&& value) {
    SetSort(std::forward<SortT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token used for pagination of results returned.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListFindingsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in the response.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListFindingsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_analyzerArn;
  bool m_analyzerArnHasBeenSet = false;

  Aws::Map<Aws::String, Criterion> m_filter;
  bool m_filterHasBeenSet = false;

  SortCriteria m_sort;
  bool m_sortHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace AccessAnalyzer
}  // namespace Aws
