﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/accessanalyzer/AccessAnalyzerRequest.h>
#include <aws/accessanalyzer/AccessAnalyzer_EXPORTS.h>
#include <aws/accessanalyzer/model/FindingStatusUpdate.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace AccessAnalyzer {
namespace Model {

/**
 * <p>Updates findings with the new values provided in the request.</p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/UpdateFindingsRequest">AWS
 * API Reference</a></p>
 */
class UpdateFindingsRequest : public AccessAnalyzerRequest {
 public:
  AWS_ACCESSANALYZER_API UpdateFindingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateFindings"; }

  AWS_ACCESSANALYZER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources">ARN
   * of the analyzer</a> that generated the findings to update.</p>
   */
  inline const Aws::String& GetAnalyzerArn() const { return m_analyzerArn; }
  inline bool AnalyzerArnHasBeenSet() const { return m_analyzerArnHasBeenSet; }
  template <typename AnalyzerArnT = Aws::String>
  void SetAnalyzerArn(AnalyzerArnT&& value) {
    m_analyzerArnHasBeenSet = true;
    m_analyzerArn = std::forward<AnalyzerArnT>(value);
  }
  template <typename AnalyzerArnT = Aws::String>
  UpdateFindingsRequest& WithAnalyzerArn(AnalyzerArnT&& value) {
    SetAnalyzerArn(std::forward<AnalyzerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The state represents the action to take to update the finding Status. Use
   * <code>ARCHIVE</code> to change an Active finding to an Archived finding. Use
   * <code>ACTIVE</code> to change an Archived finding to an Active finding.</p>
   */
  inline FindingStatusUpdate GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(FindingStatusUpdate value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline UpdateFindingsRequest& WithStatus(FindingStatusUpdate value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The IDs of the findings to update.</p>
   */
  inline const Aws::Vector<Aws::String>& GetIds() const { return m_ids; }
  inline bool IdsHasBeenSet() const { return m_idsHasBeenSet; }
  template <typename IdsT = Aws::Vector<Aws::String>>
  void SetIds(IdsT&& value) {
    m_idsHasBeenSet = true;
    m_ids = std::forward<IdsT>(value);
  }
  template <typename IdsT = Aws::Vector<Aws::String>>
  UpdateFindingsRequest& WithIds(IdsT&& value) {
    SetIds(std::forward<IdsT>(value));
    return *this;
  }
  template <typename IdsT = Aws::String>
  UpdateFindingsRequest& AddIds(IdsT&& value) {
    m_idsHasBeenSet = true;
    m_ids.emplace_back(std::forward<IdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the resource identified in the finding.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  UpdateFindingsRequest& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A client token.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateFindingsRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_analyzerArn;
  bool m_analyzerArnHasBeenSet = false;

  FindingStatusUpdate m_status{FindingStatusUpdate::NOT_SET};
  bool m_statusHasBeenSet = false;

  Aws::Vector<Aws::String> m_ids;
  bool m_idsHasBeenSet = false;

  Aws::String m_resourceArn;
  bool m_resourceArnHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace AccessAnalyzer
}  // namespace Aws
