﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/apigateway/model/CacheClusterSize.h>
#include <aws/apigateway/model/CanarySettings.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace APIGateway {
namespace Model {

/**
 * <p>Requests API Gateway to create a Stage resource.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/CreateStageRequest">AWS
 * API Reference</a></p>
 */
class CreateStageRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API CreateStageRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateStage"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The string identifier of the associated RestApi.</p>
   */
  inline const Aws::String& GetRestApiId() const { return m_restApiId; }
  inline bool RestApiIdHasBeenSet() const { return m_restApiIdHasBeenSet; }
  template <typename RestApiIdT = Aws::String>
  void SetRestApiId(RestApiIdT&& value) {
    m_restApiIdHasBeenSet = true;
    m_restApiId = std::forward<RestApiIdT>(value);
  }
  template <typename RestApiIdT = Aws::String>
  CreateStageRequest& WithRestApiId(RestApiIdT&& value) {
    SetRestApiId(std::forward<RestApiIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name for the Stage resource. Stage names can only contain alphanumeric
   * characters, hyphens, and underscores. Maximum length is 128 characters.</p>
   */
  inline const Aws::String& GetStageName() const { return m_stageName; }
  inline bool StageNameHasBeenSet() const { return m_stageNameHasBeenSet; }
  template <typename StageNameT = Aws::String>
  void SetStageName(StageNameT&& value) {
    m_stageNameHasBeenSet = true;
    m_stageName = std::forward<StageNameT>(value);
  }
  template <typename StageNameT = Aws::String>
  CreateStageRequest& WithStageName(StageNameT&& value) {
    SetStageName(std::forward<StageNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Deployment resource for the Stage resource.</p>
   */
  inline const Aws::String& GetDeploymentId() const { return m_deploymentId; }
  inline bool DeploymentIdHasBeenSet() const { return m_deploymentIdHasBeenSet; }
  template <typename DeploymentIdT = Aws::String>
  void SetDeploymentId(DeploymentIdT&& value) {
    m_deploymentIdHasBeenSet = true;
    m_deploymentId = std::forward<DeploymentIdT>(value);
  }
  template <typename DeploymentIdT = Aws::String>
  CreateStageRequest& WithDeploymentId(DeploymentIdT&& value) {
    SetDeploymentId(std::forward<DeploymentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the Stage resource.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateStageRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether cache clustering is enabled for the stage.</p>
   */
  inline bool GetCacheClusterEnabled() const { return m_cacheClusterEnabled; }
  inline bool CacheClusterEnabledHasBeenSet() const { return m_cacheClusterEnabledHasBeenSet; }
  inline void SetCacheClusterEnabled(bool value) {
    m_cacheClusterEnabledHasBeenSet = true;
    m_cacheClusterEnabled = value;
  }
  inline CreateStageRequest& WithCacheClusterEnabled(bool value) {
    SetCacheClusterEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The stage's cache capacity in GB. For more information about choosing a cache
   * size, see <a
   * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-caching.html">Enabling
   * API caching to enhance responsiveness</a>.</p>
   */
  inline CacheClusterSize GetCacheClusterSize() const { return m_cacheClusterSize; }
  inline bool CacheClusterSizeHasBeenSet() const { return m_cacheClusterSizeHasBeenSet; }
  inline void SetCacheClusterSize(CacheClusterSize value) {
    m_cacheClusterSizeHasBeenSet = true;
    m_cacheClusterSize = value;
  }
  inline CreateStageRequest& WithCacheClusterSize(CacheClusterSize value) {
    SetCacheClusterSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A map that defines the stage variables for the new Stage resource. Variable
   * names can have alphanumeric and underscore characters, and the values must match
   * <code>[A-Za-z0-9-._~:/?#&amp;=,]+</code>.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetVariables() const { return m_variables; }
  inline bool VariablesHasBeenSet() const { return m_variablesHasBeenSet; }
  template <typename VariablesT = Aws::Map<Aws::String, Aws::String>>
  void SetVariables(VariablesT&& value) {
    m_variablesHasBeenSet = true;
    m_variables = std::forward<VariablesT>(value);
  }
  template <typename VariablesT = Aws::Map<Aws::String, Aws::String>>
  CreateStageRequest& WithVariables(VariablesT&& value) {
    SetVariables(std::forward<VariablesT>(value));
    return *this;
  }
  template <typename VariablesKeyT = Aws::String, typename VariablesValueT = Aws::String>
  CreateStageRequest& AddVariables(VariablesKeyT&& key, VariablesValueT&& value) {
    m_variablesHasBeenSet = true;
    m_variables.emplace(std::forward<VariablesKeyT>(key), std::forward<VariablesValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the associated API documentation.</p>
   */
  inline const Aws::String& GetDocumentationVersion() const { return m_documentationVersion; }
  inline bool DocumentationVersionHasBeenSet() const { return m_documentationVersionHasBeenSet; }
  template <typename DocumentationVersionT = Aws::String>
  void SetDocumentationVersion(DocumentationVersionT&& value) {
    m_documentationVersionHasBeenSet = true;
    m_documentationVersion = std::forward<DocumentationVersionT>(value);
  }
  template <typename DocumentationVersionT = Aws::String>
  CreateStageRequest& WithDocumentationVersion(DocumentationVersionT&& value) {
    SetDocumentationVersion(std::forward<DocumentationVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The canary deployment settings of this stage.</p>
   */
  inline const CanarySettings& GetCanarySettings() const { return m_canarySettings; }
  inline bool CanarySettingsHasBeenSet() const { return m_canarySettingsHasBeenSet; }
  template <typename CanarySettingsT = CanarySettings>
  void SetCanarySettings(CanarySettingsT&& value) {
    m_canarySettingsHasBeenSet = true;
    m_canarySettings = std::forward<CanarySettingsT>(value);
  }
  template <typename CanarySettingsT = CanarySettings>
  CreateStageRequest& WithCanarySettings(CanarySettingsT&& value) {
    SetCanarySettings(std::forward<CanarySettingsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether active tracing with X-ray is enabled for the Stage.</p>
   */
  inline bool GetTracingEnabled() const { return m_tracingEnabled; }
  inline bool TracingEnabledHasBeenSet() const { return m_tracingEnabledHasBeenSet; }
  inline void SetTracingEnabled(bool value) {
    m_tracingEnabledHasBeenSet = true;
    m_tracingEnabled = value;
  }
  inline CreateStageRequest& WithTracingEnabled(bool value) {
    SetTracingEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The
   * tag key can be up to 128 characters and must not start with <code>aws:</code>.
   * The tag value can be up to 256 characters.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateStageRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateStageRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_restApiId;
  bool m_restApiIdHasBeenSet = false;

  Aws::String m_stageName;
  bool m_stageNameHasBeenSet = false;

  Aws::String m_deploymentId;
  bool m_deploymentIdHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  bool m_cacheClusterEnabled{false};
  bool m_cacheClusterEnabledHasBeenSet = false;

  CacheClusterSize m_cacheClusterSize{CacheClusterSize::NOT_SET};
  bool m_cacheClusterSizeHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_variables;
  bool m_variablesHasBeenSet = false;

  Aws::String m_documentationVersion;
  bool m_documentationVersionHasBeenSet = false;

  CanarySettings m_canarySettings;
  bool m_canarySettingsHasBeenSet = false;

  bool m_tracingEnabled{false};
  bool m_tracingEnabledHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
