﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/athena/AthenaRequest.h>
#include <aws/athena/Athena_EXPORTS.h>
#include <aws/athena/model/CalculationExecutionState.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Athena {
namespace Model {

/**
 */
class ListCalculationExecutionsRequest : public AthenaRequest {
 public:
  AWS_ATHENA_API ListCalculationExecutionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListCalculationExecutions"; }

  AWS_ATHENA_API Aws::String SerializePayload() const override;

  AWS_ATHENA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The session ID.</p>
   */
  inline const Aws::String& GetSessionId() const { return m_sessionId; }
  inline bool SessionIdHasBeenSet() const { return m_sessionIdHasBeenSet; }
  template <typename SessionIdT = Aws::String>
  void SetSessionId(SessionIdT&& value) {
    m_sessionIdHasBeenSet = true;
    m_sessionId = std::forward<SessionIdT>(value);
  }
  template <typename SessionIdT = Aws::String>
  ListCalculationExecutionsRequest& WithSessionId(SessionIdT&& value) {
    SetSessionId(std::forward<SessionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter for a specific calculation execution state. A description of each
   * state follows.</p> <p> <code>CREATING</code> - The calculation is in the process
   * of being created.</p> <p> <code>CREATED</code> - The calculation has been
   * created and is ready to run.</p> <p> <code>QUEUED</code> - The calculation has
   * been queued for processing.</p> <p> <code>RUNNING</code> - The calculation is
   * running.</p> <p> <code>CANCELING</code> - A request to cancel the calculation
   * has been received and the system is working to stop it.</p> <p>
   * <code>CANCELED</code> - The calculation is no longer running as the result of a
   * cancel request.</p> <p> <code>COMPLETED</code> - The calculation has completed
   * without error.</p> <p> <code>FAILED</code> - The calculation failed and is no
   * longer running.</p>
   */
  inline CalculationExecutionState GetStateFilter() const { return m_stateFilter; }
  inline bool StateFilterHasBeenSet() const { return m_stateFilterHasBeenSet; }
  inline void SetStateFilter(CalculationExecutionState value) {
    m_stateFilterHasBeenSet = true;
    m_stateFilter = value;
  }
  inline ListCalculationExecutionsRequest& WithStateFilter(CalculationExecutionState value) {
    SetStateFilter(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of calculation executions to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListCalculationExecutionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token generated by the Athena service that specifies where to continue
   * pagination if a previous request was truncated. To obtain the next set of pages,
   * pass in the <code>NextToken</code> from the response object of the previous page
   * call.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListCalculationExecutionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_sessionId;
  bool m_sessionIdHasBeenSet = false;

  CalculationExecutionState m_stateFilter{CalculationExecutionState::NOT_SET};
  bool m_stateFilterHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace Athena
}  // namespace Aws
