﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudtrail/CloudTrail_EXPORTS.h>
#include <aws/cloudtrail/model/AdvancedEventSelector.h>
#include <aws/cloudtrail/model/BillingMode.h>
#include <aws/cloudtrail/model/EventDataStoreStatus.h>
#include <aws/cloudtrail/model/FederationStatus.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace CloudTrail {
namespace Model {
class UpdateEventDataStoreResult {
 public:
  AWS_CLOUDTRAIL_API UpdateEventDataStoreResult() = default;
  AWS_CLOUDTRAIL_API UpdateEventDataStoreResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_CLOUDTRAIL_API UpdateEventDataStoreResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The ARN of the event data store.</p>
   */
  inline const Aws::String& GetEventDataStoreArn() const { return m_eventDataStoreArn; }
  template <typename EventDataStoreArnT = Aws::String>
  void SetEventDataStoreArn(EventDataStoreArnT&& value) {
    m_eventDataStoreArnHasBeenSet = true;
    m_eventDataStoreArn = std::forward<EventDataStoreArnT>(value);
  }
  template <typename EventDataStoreArnT = Aws::String>
  UpdateEventDataStoreResult& WithEventDataStoreArn(EventDataStoreArnT&& value) {
    SetEventDataStoreArn(std::forward<EventDataStoreArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the event data store.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateEventDataStoreResult& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of an event data store.</p>
   */
  inline EventDataStoreStatus GetStatus() const { return m_status; }
  inline void SetStatus(EventDataStoreStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline UpdateEventDataStoreResult& WithStatus(EventDataStoreStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The advanced event selectors that are applied to the event data store.</p>
   */
  inline const Aws::Vector<AdvancedEventSelector>& GetAdvancedEventSelectors() const { return m_advancedEventSelectors; }
  template <typename AdvancedEventSelectorsT = Aws::Vector<AdvancedEventSelector>>
  void SetAdvancedEventSelectors(AdvancedEventSelectorsT&& value) {
    m_advancedEventSelectorsHasBeenSet = true;
    m_advancedEventSelectors = std::forward<AdvancedEventSelectorsT>(value);
  }
  template <typename AdvancedEventSelectorsT = Aws::Vector<AdvancedEventSelector>>
  UpdateEventDataStoreResult& WithAdvancedEventSelectors(AdvancedEventSelectorsT&& value) {
    SetAdvancedEventSelectors(std::forward<AdvancedEventSelectorsT>(value));
    return *this;
  }
  template <typename AdvancedEventSelectorsT = AdvancedEventSelector>
  UpdateEventDataStoreResult& AddAdvancedEventSelectors(AdvancedEventSelectorsT&& value) {
    m_advancedEventSelectorsHasBeenSet = true;
    m_advancedEventSelectors.emplace_back(std::forward<AdvancedEventSelectorsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the event data store includes events from all Regions, or
   * only from the Region in which it was created.</p>
   */
  inline bool GetMultiRegionEnabled() const { return m_multiRegionEnabled; }
  inline void SetMultiRegionEnabled(bool value) {
    m_multiRegionEnabledHasBeenSet = true;
    m_multiRegionEnabled = value;
  }
  inline UpdateEventDataStoreResult& WithMultiRegionEnabled(bool value) {
    SetMultiRegionEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether an event data store is collecting logged events for an
   * organization in Organizations.</p>
   */
  inline bool GetOrganizationEnabled() const { return m_organizationEnabled; }
  inline void SetOrganizationEnabled(bool value) {
    m_organizationEnabledHasBeenSet = true;
    m_organizationEnabled = value;
  }
  inline UpdateEventDataStoreResult& WithOrganizationEnabled(bool value) {
    SetOrganizationEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The retention period, in days.</p>
   */
  inline int GetRetentionPeriod() const { return m_retentionPeriod; }
  inline void SetRetentionPeriod(int value) {
    m_retentionPeriodHasBeenSet = true;
    m_retentionPeriod = value;
  }
  inline UpdateEventDataStoreResult& WithRetentionPeriod(int value) {
    SetRetentionPeriod(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether termination protection is enabled for the event data
   * store.</p>
   */
  inline bool GetTerminationProtectionEnabled() const { return m_terminationProtectionEnabled; }
  inline void SetTerminationProtectionEnabled(bool value) {
    m_terminationProtectionEnabledHasBeenSet = true;
    m_terminationProtectionEnabled = value;
  }
  inline UpdateEventDataStoreResult& WithTerminationProtectionEnabled(bool value) {
    SetTerminationProtectionEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp that shows when an event data store was first created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedTimestamp() const { return m_createdTimestamp; }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  void SetCreatedTimestamp(CreatedTimestampT&& value) {
    m_createdTimestampHasBeenSet = true;
    m_createdTimestamp = std::forward<CreatedTimestampT>(value);
  }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  UpdateEventDataStoreResult& WithCreatedTimestamp(CreatedTimestampT&& value) {
    SetCreatedTimestamp(std::forward<CreatedTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp that shows when the event data store was last updated.
   * <code>UpdatedTimestamp</code> is always either the same or newer than the time
   * shown in <code>CreatedTimestamp</code>.</p>
   */
  inline const Aws::Utils::DateTime& GetUpdatedTimestamp() const { return m_updatedTimestamp; }
  template <typename UpdatedTimestampT = Aws::Utils::DateTime>
  void SetUpdatedTimestamp(UpdatedTimestampT&& value) {
    m_updatedTimestampHasBeenSet = true;
    m_updatedTimestamp = std::forward<UpdatedTimestampT>(value);
  }
  template <typename UpdatedTimestampT = Aws::Utils::DateTime>
  UpdateEventDataStoreResult& WithUpdatedTimestamp(UpdatedTimestampT&& value) {
    SetUpdatedTimestamp(std::forward<UpdatedTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the KMS key ID that encrypts the events delivered by CloudTrail.
   * The value is a fully specified ARN to a KMS key in the following format.</p> <p>
   * <code>arn:aws:kms:us-east-2:123456789012:key/12345678-1234-1234-1234-123456789012</code>
   * </p>
   */
  inline const Aws::String& GetKmsKeyId() const { return m_kmsKeyId; }
  template <typename KmsKeyIdT = Aws::String>
  void SetKmsKeyId(KmsKeyIdT&& value) {
    m_kmsKeyIdHasBeenSet = true;
    m_kmsKeyId = std::forward<KmsKeyIdT>(value);
  }
  template <typename KmsKeyIdT = Aws::String>
  UpdateEventDataStoreResult& WithKmsKeyId(KmsKeyIdT&& value) {
    SetKmsKeyId(std::forward<KmsKeyIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The billing mode for the event data store.</p>
   */
  inline BillingMode GetBillingMode() const { return m_billingMode; }
  inline void SetBillingMode(BillingMode value) {
    m_billingModeHasBeenSet = true;
    m_billingMode = value;
  }
  inline UpdateEventDataStoreResult& WithBillingMode(BillingMode value) {
    SetBillingMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> Indicates the <a
   * href="https://docs.aws.amazon.com/awscloudtrail/latest/userguide/query-federation.html">Lake
   * query federation</a> status. The status is <code>ENABLED</code> if Lake query
   * federation is enabled, or <code>DISABLED</code> if Lake query federation is
   * disabled. You cannot delete an event data store if the
   * <code>FederationStatus</code> is <code>ENABLED</code>. </p>
   */
  inline FederationStatus GetFederationStatus() const { return m_federationStatus; }
  inline void SetFederationStatus(FederationStatus value) {
    m_federationStatusHasBeenSet = true;
    m_federationStatus = value;
  }
  inline UpdateEventDataStoreResult& WithFederationStatus(FederationStatus value) {
    SetFederationStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> If Lake query federation is enabled, provides the ARN of the federation role
   * used to access the resources for the federated event data store. </p>
   */
  inline const Aws::String& GetFederationRoleArn() const { return m_federationRoleArn; }
  template <typename FederationRoleArnT = Aws::String>
  void SetFederationRoleArn(FederationRoleArnT&& value) {
    m_federationRoleArnHasBeenSet = true;
    m_federationRoleArn = std::forward<FederationRoleArnT>(value);
  }
  template <typename FederationRoleArnT = Aws::String>
  UpdateEventDataStoreResult& WithFederationRoleArn(FederationRoleArnT&& value) {
    SetFederationRoleArn(std::forward<FederationRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  UpdateEventDataStoreResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_eventDataStoreArn;
  bool m_eventDataStoreArnHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  EventDataStoreStatus m_status{EventDataStoreStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  Aws::Vector<AdvancedEventSelector> m_advancedEventSelectors;
  bool m_advancedEventSelectorsHasBeenSet = false;

  bool m_multiRegionEnabled{false};
  bool m_multiRegionEnabledHasBeenSet = false;

  bool m_organizationEnabled{false};
  bool m_organizationEnabledHasBeenSet = false;

  int m_retentionPeriod{0};
  bool m_retentionPeriodHasBeenSet = false;

  bool m_terminationProtectionEnabled{false};
  bool m_terminationProtectionEnabledHasBeenSet = false;

  Aws::Utils::DateTime m_createdTimestamp{};
  bool m_createdTimestampHasBeenSet = false;

  Aws::Utils::DateTime m_updatedTimestamp{};
  bool m_updatedTimestampHasBeenSet = false;

  Aws::String m_kmsKeyId;
  bool m_kmsKeyIdHasBeenSet = false;

  BillingMode m_billingMode{BillingMode::NOT_SET};
  bool m_billingModeHasBeenSet = false;

  FederationStatus m_federationStatus{FederationStatus::NOT_SET};
  bool m_federationStatusHasBeenSet = false;

  Aws::String m_federationRoleArn;
  bool m_federationRoleArnHasBeenSet = false;

  Aws::String m_requestId;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudTrail
}  // namespace Aws
