//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Info/ProjectLoadProblemDialog.cpp
//! @brief     Implements class ProjectLoadProblemDialog
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Info/ProjectLoadProblemDialog.h"
#include "GUI/Support/Util/DesignerHelper.h"
#include "GUI/Support/Util/Path.h"
#include <QGridLayout>
#include <QLabel>
#include <QListWidget>
#include <QPushButton>

namespace {

const int top_panel_height = 80;
}

ProjectLoadProblemDialog::ProjectLoadProblemDialog(QWidget* parent, const QStringList& details,
                                                   QString documentVersion)
    : QDialog(parent)
    , m_projectDocumentVersion(std::move(documentVersion))
{
    setMinimumSize(256, 256);
    resize(520, 620);
    setWindowTitle("Problems encountered while loading project");
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    auto* mainLayout = new QVBoxLayout;

    QFont titleFont;
    titleFont.setPointSize(DesignerHelper::getSectionFontSize());
    titleFont.setBold(true);

    const auto addHeadline = [&](const QString& text) {
        auto* label = new QLabel(text, this);
        label->setFont(titleFont);
        mainLayout->addWidget(label);
    };

    // -- warning icon + info
    auto* topPanelLayout = new QHBoxLayout;
    auto* messageLabel = new QLabel("Some parts of the project were not loaded correctly.");
    messageLabel->setFont(titleFont);
    messageLabel->setWordWrap(true);

    topPanelLayout->addWidget(createWarningWidget());
    topPanelLayout->addWidget(messageLabel);
    topPanelLayout->setContentsMargins(0, 0, 0, 0);
    mainLayout->addLayout(topPanelLayout);

    // -- explanations
    addHeadline("Why did this happen?");
    auto* explanationLabel = new QLabel(explanationText());
    explanationLabel->setWordWrap(true);
    mainLayout->addWidget(explanationLabel);

    addHeadline("What to do?");
    auto* adviceLabel =
        new QLabel("Check parameters of your items and re-enter uninitialized values. "
                   "Use detailed log below to get a hint what went wrong. "
                   "After that, save your project and work as normal.");
    adviceLabel->setWordWrap(true);
    mainLayout->addWidget(adviceLabel);

    // -- details view
    addHeadline("Details:");
    auto* detailsWidget = new QListWidget;
    detailsWidget->setWordWrap(true);
    detailsWidget->setAlternatingRowColors(true);
    detailsWidget->addItems(details);
    mainLayout->addWidget(detailsWidget);

    // -- buttons
    mainLayout->addLayout(buttonLayout());

    setLayout(mainLayout);
    setAttribute(Qt::WA_DeleteOnClose, true);
}

QWidget* ProjectLoadProblemDialog::createWarningWidget()
{
    auto* warningLabel = new QLabel;
    warningLabel->setPixmap(QPixmap(":/images/warning_64x64.png"));

    auto* warningWidget = new QWidget;
    warningWidget->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    warningWidget->resize(top_panel_height, top_panel_height);
    auto* warningWidgetLayout = new QHBoxLayout;
    warningWidgetLayout->addWidget(warningLabel);
    warningWidget->setLayout(warningWidgetLayout);

    return warningWidget;
}

QLayout* ProjectLoadProblemDialog::buttonLayout()
{
    auto* button = new QPushButton("Close", this);
    button->setAutoDefault(false);
    connect(button, &QPushButton::clicked, this, &ProjectLoadProblemDialog::close);

    auto* result = new QHBoxLayout;
    result->addStretch(3);
    result->setContentsMargins(0, 0, 0, 0);
    result->addWidget(button);

    return result;
}

//! Returns explanations what went wrong.
QString ProjectLoadProblemDialog::explanationText() const
{
    if (m_projectDocumentVersion != GUI::Base::Path::getBornAgainVersionString()) {
        return QString(
                   "Given project was created using BornAgain version %1 "
                   " which is different from version %2 you are currently using. "
                   "At the moment we provide only limited support for import from older versions.")
            .arg(m_projectDocumentVersion)
            .arg(GUI::Base::Path::getBornAgainVersionString());
    }

    return QString("Given project was created using BornAgain version %1 "
                   "which is the same as the current version of the framework. "
                   "Strangely enough, some parts were not loaded correctly due to format mismatch. "
                   "Please contact the developers.")
        .arg(m_projectDocumentVersion);
}
