"""Tests for server wrapper and helper functions"""

from juju.providers.openstack.machine import (
    get_server_addresses,
    get_server_status,
    )

from juju.lib.testing import TestCase


class GetServerStatusTest(TestCase):
    """Tests for mapping of Nova status names to EC2 style names"""

    def test_build_schedualing(self):
        self.assertEqual("pending",
            get_server_status({u'status': u'BUILD(scheduling)'}))

    def test_build_spawning(self):
        self.assertEqual("pending",
            get_server_status({u'status': u'BUILD(spawning)'}))

    def test_active(self):
        self.assertEqual("running",
            get_server_status({u'status': u'ACTIVE'}))

    def test_no_status(self):
        self.assertEqual("pending",
            get_server_status({}))

    def test_mystery_status(self):
        self.assertEqual("unknown",
            get_server_status({u'status': u'NEVER_BEFORE_SEEN_MYSTERY'}))


class GetServerAddressesTest(TestCase):
    """Tests for deriving a public and private address from Nova dict"""

    def test_missing(self):
        self.assertEqual((None, None), get_server_addresses({}))

    def test_empty(self):
        self.assertEqual((None, None),
            get_server_addresses({u'addresses': {}}))

    def test_private_only(self):
        self.assertEqual(("127.0.0.4", None),
            get_server_addresses({u'addresses': {
                "private": [{"addr": "127.0.0.4", "version": 4}],
            }}))

    def test_private_plus(self):
        self.assertEqual(("127.0.0.4", "8.8.4.4"),
            get_server_addresses({u'addresses': {
                "private": [
                    {"addr": "127.0.0.4", "version": 4},
                    {"addr": "8.8.4.4", "version": 4},
                ],
            }}))

    def test_public_only(self):
        self.assertEqual((None, "8.8.8.8"),
            get_server_addresses({u'addresses': {
                "public": [{"addr": "8.8.8.8", "version": 4}],
            }}))

    def test_public_and_private(self):
        self.assertEqual(("127.0.0.4", "8.8.8.8"),
            get_server_addresses({u'addresses': {
                "public": [{"addr": "8.8.8.8", "version": 4}],
                "private": [{"addr": "127.0.0.4", "version": 4}],
            }}))

    def test_public_and_private_plus(self):
        self.assertEqual(("127.0.0.4", "8.8.8.8"),
            get_server_addresses({u'addresses': {
                "public": [{"addr": "8.8.8.8", "version": 4}],
                "private": [
                    {"addr": "127.0.0.4", "version": 4},
                    {"addr": "8.8.4.4", "version": 4},
                ],
            }}))

    def test_custom_only(self):
        self.assertEqual(("127.0.0.2", None),
            get_server_addresses({u'addresses': {
                "special": [{"addr": "127.0.0.2", "version": 4}],
            }}))

    def test_custom_plus(self):
        self.assertEqual(("127.0.0.2", "8.8.4.4"),
            get_server_addresses({u'addresses': {
                "special": [
                    {"addr": "127.0.0.2", "version": 4},
                    {"addr": "8.8.4.4", "version": 4},
                ],
            }}))

    def test_custom_and_private(self):
        self.assertEqual(("127.0.0.4", None),
            get_server_addresses({u'addresses': {
                "special": [{"addr": "127.0.0.2", "version": 4}],
                "private": [{"addr": "127.0.0.4", "version": 4}],
            }}))

    def test_custom_and_public(self):
        self.assertEqual(("127.0.0.2", "8.8.8.8"),
            get_server_addresses({u'addresses': {
                "special": [{"addr": "127.0.0.2", "version": 4}],
                "public": [{"addr": "8.8.8.8", "version": 4}],
            }}))
