/* -*- mode: c; c-basic-offset: 4; indent-tabs-mode: nil -*- */
/*
 * Copyright (C) 2013 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Pawel Stolowski <pawel.stolowski@canonical.com>
 */

#include "SmartScopesPreviewParser.h"
#include "Utils.h"
#include <QJsonDocument>
#include <QJsonArray>
#include <QDebug>


UnityPreview* SmartScopesPreviewParser::parse(const QString &jsonText)
{
    const QJsonDocument doc = QJsonDocument::fromJson(jsonText.toUtf8());
    const QJsonObject root = doc.object();

    auto it = root.find("renderer_name");
    if (it == root.end())
    {
        qWarning() << "Missing 'renderer_name' element";
        return nullptr;
    }

    auto renderer = it.value().toString();
    if (renderer == "preview-music")
    {
        return parseMusicPreview(root);
    }
    else if (renderer == "preview-generic")
    {
        return parseGenericPreview(root);
    }

    qWarning() << "Unsupported renderer" << renderer << ", using generic preview";
    return parseGenericPreview(root);
}

UnityPreview* SmartScopesPreviewParser::parseGenericPreview(const QJsonObject& root)
{
    QString title, subtitle, description, attribution;
    GIcon *icon;
    getBaseAttributes(root, title, subtitle, description, &icon, attribution);

    auto preview = unity_generic_preview_new(title.toUtf8().data(), description.toUtf8().data(), icon);
    g_object_unref(icon);
    unity_preview_set_subtitle(UNITY_PREVIEW(preview), subtitle.toUtf8().data());

    getInfoHints(root, UNITY_PREVIEW(preview));
    getActions(root, UNITY_PREVIEW(preview));

    return UNITY_PREVIEW(preview);
}

UnityPreview* SmartScopesPreviewParser::parseMusicPreview(const QJsonObject& root)
{
    QString title, subtitle, description, attribution;
    GIcon *icon;
    getBaseAttributes(root, title, subtitle, description, &icon, attribution);

    auto preview = unity_music_preview_new(title.toUtf8().data(), subtitle.toUtf8().data(), icon);
    g_object_unref(icon);

    getInfoHints(root, UNITY_PREVIEW(preview));
    getActions(root, UNITY_PREVIEW(preview));
    getTracks(root, preview);

    return UNITY_PREVIEW(preview);
}

void SmartScopesPreviewParser::getBaseAttributes(const QJsonObject& root, QString &title, QString& subtitle, QString &description, GIcon **icon, QString &attribution)
{
    auto it = root.find("title");
    if (it != root.end())
        title = it.value().toString();

    it = root.find("subtitle");
    if (it != root.end())
        subtitle = it.value().toString();

    it = root.find("description_html");
    if (it != root.end())
    {
        description = it.value().toString();
    }
    else
    {
        it = root.find("description");
        if (it != root.end())
            description = it.value().toString();
    }

    *icon = nullptr;
    it = root.find("image_hint");
    if (it != root.end())
    {
        auto iconUri = it.value().toString();
        auto iconFile = g_file_new_for_uri(iconUri.toUtf8().data());
        *icon = g_file_icon_new(iconFile);
        g_object_unref(iconFile);
    }

    it = root.find("attribution");
    if (it != root.end())
    {
        attribution = it.value().toString();
        if (!attribution.isEmpty())
        {
            auto escaped = g_markup_escape_text(attribution.toUtf8().data(), -1); //FIXME: is there a more Qt'ish way to do that?
            attribution = "\n──────────────\n<small>" + QString(escaped) + "</small>";
            g_free(escaped);
        }
    }
}

void SmartScopesPreviewParser::getInfoHints(const QJsonObject& root, UnityPreview *preview)
{
    auto it = root.find("info");
    if (it != root.end())
    {
        const QJsonArray hints = it.value().toArray();
        for (auto hintIt = hints.begin(); hintIt != hints.end(); hintIt++)
        {
            const QJsonObject hintObj = (*hintIt).toObject();

            QString id, displayName, value;
            GIcon *icon = nullptr;

            auto hintPropIt = hintObj.find("id");
            if (hintPropIt == hintObj.end())
                continue;
            id = hintPropIt.value().toString();

            hintPropIt = hintObj.find("display_name");
            if (hintPropIt == hintObj.end())
                continue;
            displayName = hintPropIt.value().toString();

            hintPropIt = hintObj.find("value");
            if (hintPropIt == hintObj.end())
                continue;
            value = hintPropIt.value().toString();

            hintPropIt = hintObj.find("icon_hint");
            if (hintPropIt != hintObj.end())
                icon = gfileIconFromUri(hintPropIt.value().toString());

            auto info = unity_info_hint_new(id.toUtf8().data(), displayName.toUtf8().data(), icon, value.toUtf8().data());
            unity_preview_add_info(preview, info);
        }
    }
}

void SmartScopesPreviewParser::getActions(const QJsonObject& root, UnityPreview *preview)
{
    auto it = root.find("actions");
    if (it != root.end())
    {
        const QJsonArray actions = it.value().toArray();
        for (auto actionIt = actions.begin(); actionIt != actions.end(); actionIt++)
        {
            const QJsonObject actionObj = (*actionIt).toObject();

            QString uri, displayName;
            GIcon *icon = nullptr;

            auto actionPropIt = actionObj.find("activation_uri");
            if (actionPropIt == actionObj.end())
                continue;
            uri = actionPropIt.value().toString();

            actionPropIt = actionObj.find("display_name");
            if (actionPropIt == actionObj.end())
                continue;
            displayName = actionPropIt.value().toString();

            actionPropIt = actionObj.find("icon_hint");
            if (actionPropIt != actionObj.end())
                icon = gfileIconFromUri(actionPropIt.value().toString());

            auto action = unity_preview_action_new(uri.toUtf8().data(), displayName.toUtf8().data(), icon);
            actionPropIt = actionObj.find("extra_text");
            if (actionPropIt != actionObj.end())
                unity_preview_action_set_extra_text(action, actionPropIt.value().toString().toUtf8().data());

            unity_preview_add_action(preview, action);
        }
    }
}

void SmartScopesPreviewParser::getTracks(const QJsonObject& root, UnityMusicPreview *preview)
{
    auto it = root.find("tracks");
    if (it != root.end())
    {
        const QJsonArray tracks = it.value().toArray();
        for (auto trackIt = tracks.begin(); trackIt != tracks.end(); trackIt++)
        {
            const QJsonObject trackObj = (*trackIt).toObject();

            QString uri, title;
            int trackNo, length;

            auto trackPropIt = trackObj.find("uri");
            if (trackPropIt == trackObj.end())
                continue;
            uri = trackPropIt.value().toString();

            trackPropIt = trackObj.find("title");
            if (trackPropIt == trackObj.end())
                continue;
            title = trackPropIt.value().toString();

            trackPropIt = trackObj.find("track_no");
            if (trackPropIt == trackObj.end())
                continue;
            trackNo = trackPropIt.value().toDouble();

            trackPropIt = trackObj.find("length");
            if (trackPropIt == trackObj.end())
                continue;
            length = trackPropIt.value().toDouble();

            auto trackData = unity_track_metadata_new();
            unity_track_metadata_set_uri(trackData, uri.toUtf8().data());
            unity_track_metadata_set_title(trackData, title.toUtf8().data());
            unity_track_metadata_set_track_no(trackData, trackNo);
            unity_track_metadata_set_length(trackData, length);
            
            unity_music_preview_add_track(preview, trackData);
        }
    }
}
